(function () {
    'use strict';
    
    window.admin = {
        parts: {},
        pages: {}
    };
})();
(function () {
    'use strict';

    /**
     * Modern Card List Component
     * Replaces FooTable functionality with enhanced UX
     */
    class CardList {
        constructor(container) {
            this.container = container;
            this.checkboxes = container.querySelectorAll('.card-checkbox');
            this.selectAllCheckbox = document.querySelector('#select_all');
            this.cardSelectAllBtn = document.querySelector('#shared-select-all');
            this.batchBar = container.querySelector('.card-list-batch-bar');
            this.batchCount = container.querySelector('.batch-count');
            this.cards = container.querySelectorAll('.card-list-item');

            this.selectedItems = new Set();
            this.isInitialized = false;

            this.init();
        }

        init() {
            if (this.isInitialized) return;

            this.setupEventListeners();
            this.setupKeyboardNavigation();
            this.updateBatchState();
            this.isInitialized = true;

            // Add loading state removal
            this.container.classList.remove('card-list-loading');
        }

        setupEventListeners() {
            // Individual checkbox handling
            this.checkboxes.forEach(checkbox => {
                checkbox.addEventListener('change', (e) => {
                    this.handleCheckboxChange(e.target);
                });

                // Add card click to select
                const card = checkbox.closest('.card-list-item');
                if (card) {
                    card.addEventListener('click', (e) => {
                        // Only trigger if not clicking on buttons or links
                        if (!e.target.closest('.btn') && !e.target.closest('a') &&
                            !e.target.closest('.card-checkbox-container')) {
                            checkbox.checked = !checkbox.checked;
                            this.handleCheckboxChange(checkbox);
                        }
                    });
                }
            });

            // Select all checkbox (traditional table header)
            if (this.selectAllCheckbox) {
                this.selectAllCheckbox.addEventListener('change', (e) => {
                    this.handleSelectAll(e.target.checked);
                });
            }

            // Note: Card view select all button is now handled by SharedControlBar component
            // No event listener needed here as SharedControlBar handles the #shared-select-all button

            // Batch operations
            this.setupBatchOperations();

            // Add hover effects
            this.setupHoverEffects();
        }

        setupKeyboardNavigation() {
            document.addEventListener('keydown', (e) => {
                if (!this.isCardListFocused()) return;

                switch (e.key) {
                    case 'Escape':
                        this.clearSelection();
                        break;
                    case 'a':
                        if (e.ctrlKey || e.metaKey) {
                            e.preventDefault();
                            this.selectAll();
                        }
                        break;
                }
            });
        }

        setupHoverEffects() {
            this.cards.forEach(card => {
                card.addEventListener('mouseenter', () => {
                    card.classList.add('card-hover');
                });

                card.addEventListener('mouseleave', () => {
                    card.classList.remove('card-hover');
                });
            });
        }

        setupBatchOperations() {
            // Add batch operation buttons dynamically based on existing batch operations
            const existingBatchForm = document.querySelector('form[action*="batch"]');
            if (existingBatchForm && this.batchBar) {
                const batchActions = this.batchBar.querySelector('.batch-actions');
                if (batchActions) {
                    // Copy existing batch buttons and modify them
                    const existingButtons = existingBatchForm.querySelectorAll('.btn');
                    existingButtons.forEach(btn => {
                        if (btn.type === 'submit') {
                            const newBtn = btn.cloneNode(true);
                            newBtn.classList.add('btn-sm');
                            newBtn.addEventListener('click', (e) => {
                                e.preventDefault();
                                this.executeBatchOperation(btn.name, btn.value);
                            });
                            batchActions.appendChild(newBtn);
                        }
                    });
                }
            }
        }

        handleCheckboxChange(checkbox) {
            const card = checkbox.closest('.card-list-item');
            const value = checkbox.value;

            // Add selection animation
            card.classList.add('selecting');
            setTimeout(() => card.classList.remove('selecting'), 150);

            if (checkbox.checked) {
                this.selectedItems.add(value);
                card.classList.add('selected');
            } else {
                this.selectedItems.delete(value);
                card.classList.remove('selected');
            }

            this.updateBatchState();
            this.updateSelectAllState();
        }

        handleSelectAll(checked) {
            this.checkboxes.forEach(checkbox => {
                const wasChecked = checkbox.checked;
                checkbox.checked = checked;

                if (wasChecked !== checked) {
                    this.handleCheckboxChange(checkbox);
                }
            });
        }

        updateBatchState() {
            const selectedCount = this.selectedItems.size;

            if (selectedCount > 0) {
                this.showBatchBar();
                this.updateBatchCount(selectedCount);
                this.container.classList.add('batch-mode');
            } else {
                this.hideBatchBar();
                this.container.classList.remove('batch-mode');
            }
        }

        updateBatchCount(count) {
            if (this.batchCount) {
                const text = count === 1 ?
                    `${count} item selected` :
                    `${count} items selected`;
                this.batchCount.textContent = text;
            }
        }

        updateSelectAllState() {
            const totalCheckboxes = this.checkboxes.length;
            const selectedCount = this.selectedItems.size;

            // Update traditional select all checkbox
            if (this.selectAllCheckbox) {
                if (selectedCount === 0) {
                    this.selectAllCheckbox.checked = false;
                    this.selectAllCheckbox.indeterminate = false;
                } else if (selectedCount === totalCheckboxes) {
                    this.selectAllCheckbox.checked = true;
                    this.selectAllCheckbox.indeterminate = false;
                } else {
                    this.selectAllCheckbox.checked = false;
                    this.selectAllCheckbox.indeterminate = true;
                }
            }

            // Update card select all button
            if (this.cardSelectAllBtn) {
                const icon = this.cardSelectAllBtn.querySelector('i');
                const text = this.cardSelectAllBtn.querySelector('span');

                if (selectedCount === 0) {
                    // None selected
                    this.cardSelectAllBtn.classList.remove('active');
                    if (icon) icon.className = 'fa fa-square-o';
                    if (text) text.textContent = 'Select All';
                } else if (selectedCount === totalCheckboxes) {
                    // All selected
                    this.cardSelectAllBtn.classList.add('active');
                    if (icon) icon.className = 'fa fa-check-square-o';
                    if (text) text.textContent = 'Deselect All';
                } else {
                    // Some selected
                    this.cardSelectAllBtn.classList.add('active');
                    if (icon) icon.className = 'fa fa-minus-square-o';
                    if (text) text.textContent = `Deselect (${selectedCount})`;
                }
            }
        }

        showBatchBar() {
            if (this.batchBar) {
                this.batchBar.style.display = 'block';
                // Trigger reflow for animation
                this.batchBar.offsetHeight;
                this.batchBar.classList.add('show');
            }
        }

        hideBatchBar() {
            if (this.batchBar) {
                this.batchBar.classList.remove('show');
                setTimeout(() => {
                    this.batchBar.style.display = 'none';
                }, 300);
            }
        }

        selectAll() {
            if (this.selectAllCheckbox) {
                this.selectAllCheckbox.checked = true;
                this.handleSelectAll(true);
            }
        }

        clearSelection() {
            this.selectedItems.clear();
            this.checkboxes.forEach(checkbox => {
                checkbox.checked = false;
                const card = checkbox.closest('.card-list-item');
                card.classList.remove('selected');
            });
            this.updateBatchState();
            this.updateSelectAllState();
        }

        executeBatchOperation(action, value) {
            if (this.selectedItems.size === 0) {
                return;
            }

            // Create form data
            const formData = new FormData();
            formData.append(action, value);

            this.selectedItems.forEach(item => {
                formData.append('batch[]', item);
            });

            // Add loading state
            this.container.classList.add('card-list-loading');

            // Submit the form (using existing form submission logic)
            const existingForm = document.querySelector('form[action*="batch"]');
            if (existingForm) {
                // Clear existing batch checkboxes
                existingForm.querySelectorAll('input[name="batch[]"]').forEach(input => {
                    input.remove();
                });

                // Add selected items to form
                this.selectedItems.forEach(item => {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = 'batch[]';
                    input.value = item;
                    existingForm.appendChild(input);
                });

                // Add action button
                const actionBtn = document.createElement('input');
                actionBtn.type = 'hidden';
                actionBtn.name = action;
                actionBtn.value = value;
                existingForm.appendChild(actionBtn);

                // Submit form
                existingForm.submit();
            }
        }

        isCardListFocused() {
            const activeElement = document.activeElement;
            return this.container.contains(activeElement) ||
                   activeElement === document.body;
        }

        // Public API
        getSelectedItems() {
            return Array.from(this.selectedItems);
        }

        setSelectedItems(items) {
            this.clearSelection();
            items.forEach(item => {
                const checkbox = this.container.querySelector(`input[value="${item}"]`);
                if (checkbox) {
                    checkbox.checked = true;
                    this.handleCheckboxChange(checkbox);
                }
            });
        }

        refresh() {
            this.clearSelection();
            this.checkboxes = this.container.querySelectorAll('.card-checkbox');
            this.cardSelectAllBtn = this.container.querySelector('#card-select-all');
            this.cards = this.container.querySelectorAll('.card-list-item');
            this.setupEventListeners();
        }
    }

    /**
     * Auto-initialize card lists on page load
     */
    function initializeCardLists() {
        const cardLists = document.querySelectorAll('.card-list');
        cardLists.forEach(container => {
            if (!container.cardListInstance) {
                container.cardListInstance = new CardList(container);
            }
        });
    }

    /**
     * Enhanced animations and transitions
     */
    function setupCardAnimations() {
        // Intersection Observer for entrance animations
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-in');
                        observer.unobserve(entry.target);
                    }
                });
            }, {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            });

            document.querySelectorAll('.card-list-item').forEach(card => {
                observer.observe(card);
            });
        }
    }

    /**
     * Smooth scrolling for batch operations
     */
    function setupSmoothScrolling() {
        const batchBars = document.querySelectorAll('.card-list-batch-bar');
        batchBars.forEach(bar => {
            if (bar.classList.contains('show')) {
                bar.scrollIntoView({
                    behavior: 'smooth',
                    block: 'nearest'
                });
            }
        });
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            initializeCardLists();
            setupCardAnimations();
        });
    } else {
        initializeCardLists();
        setupCardAnimations();
    }

    // Re-initialize on dynamic content changes
    const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1) { // Element node
                        const cardList = node.querySelector ? node.querySelector('.card-list') : null;
                        if (cardList && !cardList.cardListInstance) {
                            cardList.cardListInstance = new CardList(cardList);
                        }
                    }
                });
            }
        });
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });

    // Export for external use
    window.CardList = CardList;

})();
(function () {
    'use strict';

    /**
     * File Information Panel Component
     * Provides detailed file information in a sliding side panel
     */
    class FileInfoPanel {
        constructor() {
            this.panel = document.getElementById('file-info-panel');
            this.overlay = document.getElementById('info-panel-overlay');
            this.loadingElement = document.getElementById('file-info-loading');
            this.dataElement = document.getElementById('file-info-data');
            this.closeButton = document.getElementById('close-info-panel');

            this.isOpen = false;
            this.currentFileId = null;

            this.init();
        }

        init() {
            if (!this.panel || !this.overlay) return;

            this.setupEventListeners();
        }

        setupEventListeners() {
            // File info button clicks
            document.addEventListener('click', (e) => {
                if (e.target.closest('.file-info-btn')) {
                    e.preventDefault();
                    e.stopPropagation();

                    const button = e.target.closest('.file-info-btn');
                    const fileId = button.getAttribute('data-file-id');
                    this.openPanel(fileId);
                }
            });

            // Close button
            if (this.closeButton) {
                this.closeButton.addEventListener('click', () => {
                    this.closePanel();
                });
            }

            // Overlay click to close
            if (this.overlay) {
                this.overlay.addEventListener('click', () => {
                    this.closePanel();
                });
            }

            // Escape key to close
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape' && this.isOpen) {
                    this.closePanel();
                }
            });
        }

        openPanel(fileId) {
            if (!fileId) return;

            this.currentFileId = fileId;
            this.isOpen = true;

            // Show loading state
            this.showLoading();

            // Show overlay and panel
            this.overlay.style.display = 'block';
            this.panel.style.display = 'flex';

            // Trigger animations
            setTimeout(() => {
                this.overlay.classList.add('show');
                this.panel.classList.add('show');
            }, 10);

            // Load file information
            this.loadFileInfo(fileId);
        }

        closePanel() {
            this.isOpen = false;

            // Hide panel with animation
            this.overlay.classList.remove('show');
            this.panel.classList.remove('show');

            // Hide elements after animation
            setTimeout(() => {
                this.overlay.style.display = 'none';
                this.panel.style.display = 'none';
            }, 300);
        }

        showLoading() {
            this.loadingElement.style.display = 'flex';
            this.dataElement.style.display = 'none';
        }

        showData() {
            this.loadingElement.style.display = 'none';
            this.dataElement.style.display = 'block';
        }

        showError(message) {
            this.dataElement.innerHTML = `
                <div style="padding: 40px 24px; text-align: center; color: var(--color-danger);">
                    <i class="fa fa-exclamation-triangle" style="font-size: 2rem; margin-bottom: 16px;"></i>
                    <p style="margin: 0;">${message}</p>
                </div>
            `;
            this.showData();
        }

        loadFileInfo(fileId) {
            // Use jQuery for AJAX to maintain compatibility
            $.ajax({
                url: json_strings.uri.base + 'process.php',
                method: 'GET',
                data: {
                    do: 'get_file_info',
                    file_id: fileId
                },
                dataType: 'json'
            }).done((response) => {
                if (response.success && response.file) {
                    this.renderFileInfo(response.file);
                } else {
                    this.showError(response.error || 'Error loading file information');
                }
            }).fail(() => {
                this.showError('Failed to load file information');
            });
        }

        renderFileInfo(file) {
            const html = this.buildFileInfoHTML(file);
            this.dataElement.innerHTML = html;
            this.showData();
        }

        buildFileInfoHTML(file) {
            let html = '';

            // File preview section
            html += '<div class="file-preview-section">';
            if (file.is_image && file.thumbnail) {
                html += `<img src="${file.thumbnail}" alt="${file.title}" class="file-thumbnail-large">`;
            } else {
                const colorClass = this.getExtensionColorClass(file.extension);
                html += `<div class="file-icon-large ${colorClass}">${file.extension ? file.extension.toUpperCase() : 'FILE'}</div>`;
            }
            html += `<h4 class="file-title">${file.title}</h4>`;
            html += '</div>';

            // File details section
            html += '<div class="file-details-section">';

            // Basic information
            html += '<div class="detail-group">';
            html += '<div class="detail-group-title">Basic Information</div>';

            html += '<div class="detail-item">';
            html += '<span class="detail-label">Original filename</span>';
            html += `<span class="detail-value">${file.filename_original || file.title}</span>`;
            html += '</div>';

            if (file.description) {
                html += '<div class="detail-item">';
                html += '<span class="detail-label">Description</span>';
                html += `<span class="detail-value${file.description.length > 100 ? ' truncate' : ''}">${file.description}</span>`;
                html += '</div>';
            }

            html += '<div class="detail-item">';
            html += '<span class="detail-label">File size</span>';
            html += `<span class="detail-value">${file.size_formatted}</span>`;
            html += '</div>';

            html += '<div class="detail-item">';
            html += '<span class="detail-label">File type</span>';
            html += `<span class="detail-value">${file.extension ? file.extension.toUpperCase() + ' file' : 'Unknown'}</span>`;
            html += '</div>';

            html += '<div class="detail-item">';
            html += '<span class="detail-label">Upload date</span>';
            html += `<span class="detail-value">${file.uploaded_date}</span>`;
            html += '</div>';

            if (file.uploaded_by) {
                html += '<div class="detail-item">';
                html += '<span class="detail-label">Uploaded by</span>';
                html += `<span class="detail-value">${file.uploaded_by}</span>`;
                html += '</div>';
            }
            html += '</div>';

            // Image metadata (for images)
            if (file.image_metadata) {
                html += '<div class="detail-group">';
                html += '<div class="detail-group-title">Image Information</div>';

                if (file.image_metadata.width && file.image_metadata.height) {
                    html += '<div class="detail-item">';
                    html += '<span class="detail-label">Dimensions</span>';
                    html += `<span class="detail-value">${file.image_metadata.width} × ${file.image_metadata.height} pixels</span>`;
                    html += '</div>';
                }

                if (file.image_metadata.exif) {
                    const exif = file.image_metadata.exif;
                    if (exif.Make || exif.Model) {
                        html += '<div class="detail-item">';
                        html += '<span class="detail-label">Camera</span>';
                        html += `<span class="detail-value">${exif.Make || ''} ${exif.Model || ''}</span>`;
                        html += '</div>';
                    }
                    if (exif.DateTime) {
                        html += '<div class="detail-item">';
                        html += '<span class="detail-label">Taken on</span>';
                        html += `<span class="detail-value">${exif.DateTime}</span>`;
                        html += '</div>';
                    }
                    if (exif.ExposureTime) {
                        html += '<div class="detail-item">';
                        html += '<span class="detail-label">Exposure</span>';
                        html += `<span class="detail-value">${exif.ExposureTime}</span>`;
                        html += '</div>';
                    }
                    if (exif.FNumber) {
                        html += '<div class="detail-item">';
                        html += '<span class="detail-label">F-stop</span>';
                        html += `<span class="detail-value">f/${exif.FNumber}</span>`;
                        html += '</div>';
                    }
                    if (exif.ISOSpeedRatings) {
                        html += '<div class="detail-item">';
                        html += '<span class="detail-label">ISO</span>';
                        html += `<span class="detail-value">${exif.ISOSpeedRatings}</span>`;
                        html += '</div>';
                    }
                }
                html += '</div>';
            }

            // Status information
            html += '<div class="detail-group">';
            html += '<div class="detail-group-title">Status & Permissions</div>';

            html += '<div class="detail-item">';
            html += '<span class="detail-label">Privacy</span>';
            const publicStatus = file.public == 1 ? 'success' : 'warning';
            const publicText = file.public == 1 ? 'Public' : 'Private';
            const publicIcon = file.public == 1 ? 'fa-globe' : 'fa-lock';
            html += `<span class="detail-value"><span class="status-badge ${publicStatus}"><i class="fa ${publicIcon}"></i> ${publicText}</span></span>`;
            html += '</div>';

            if (file.public == 1 && file.public_url) {
                html += '<div class="detail-item">';
                html += '<span class="detail-label">Public URL</span>';
                html += `<span class="detail-value"><a href="${file.public_url}" target="_blank" class="text-primary"><i class="fa fa-external-link"></i> View public page</a></span>`;
                html += '</div>';
            }

            if (file.expires == 1) {
                html += '<div class="detail-item">';
                html += '<span class="detail-label">Expiry</span>';
                const isExpired = file.expired;
                const expiryStatus = isExpired ? 'danger' : (file.days_until_expiry <= 7 ? 'warning' : 'success');
                const expiryIcon = isExpired ? 'fa-clock-o' : 'fa-calendar-check-o';
                let expiryText = file.expiry_date_formatted || 'Set to expire';
                if (!isExpired && file.days_until_expiry !== undefined) {
                    expiryText += ` (${file.days_until_expiry} days)`;
                }
                html += `<span class="detail-value"><span class="status-badge ${expiryStatus}"><i class="fa ${expiryIcon}"></i> ${expiryText}</span></span>`;
                html += '</div>';
            }

            if (file.download_count !== undefined) {
                html += '<div class="detail-item">';
                html += '<span class="detail-label">Downloads</span>';
                html += `<span class="detail-value"><span class="status-badge info"><i class="fa fa-download"></i> ${file.download_count}</span></span>`;
                html += '</div>';
            }

            html += '</div>';

            // Categories (if available)
            if (file.categories && file.categories.length > 0) {
                html += '<div class="detail-group">';
                html += '<div class="detail-group-title">Categories</div>';
                html += '<div class="detail-item">';
                html += '<div class="detail-value">';
                file.categories.forEach(category => {
                    html += `<span class="status-badge info" style="margin: 2px;"><i class="fa fa-tag"></i> ${category.name}</span>`;
                });
                html += '</div>';
                html += '</div>';
                html += '</div>';
            }

            // Assignments (if available and user has permission)
            if (file.assignments && (file.assignments.clients.length > 0 || file.assignments.groups.length > 0)) {
                html += '<div class="detail-group">';
                html += '<div class="detail-group-title">Assignments</div>';

                if (file.assignments.clients.length > 0) {
                    html += '<div class="detail-item">';
                    html += '<span class="detail-label">Clients</span>';
                    html += `<span class="detail-value">${file.assignments.clients.length} client(s)</span>`;
                    html += '</div>';
                }

                if (file.assignments.groups.length > 0) {
                    html += '<div class="detail-item">';
                    html += '<span class="detail-label">Groups</span>';
                    html += `<span class="detail-value">${file.assignments.groups.length} group(s)</span>`;
                    html += '</div>';
                }

                html += '</div>';
            }

            html += '</div>';

            // Actions section
            html += '<div class="file-actions-section">';
            html += '<div class="action-buttons">';

            if (file.edit_url) {
                html += `<a href="${file.edit_url}" class="btn btn-primary"><i class="fa fa-edit"></i> Edit</a>`;
            }

            if (file.download_url) {
                html += `<a href="${file.download_url}" class="btn btn-success" target="_blank"><i class="fa fa-download"></i> Download</a>`;
            }

            html += '</div>';
            html += '</div>';

            return html;
        }

        getExtensionColorClass(extension) {
            if (!extension) return 'badge-default';

            const ext = extension.toUpperCase();
            const colorMap = {
                'PDF': 'badge-red',
                'DOC': 'badge-blue', 'DOCX': 'badge-blue',
                'XLS': 'badge-green', 'XLSX': 'badge-green',
                'PPT': 'badge-orange', 'PPTX': 'badge-orange',
                'ZIP': 'badge-purple', 'RAR': 'badge-purple', '7Z': 'badge-purple',
                'MP3': 'badge-pink', 'WAV': 'badge-pink', 'FLAC': 'badge-pink',
                'MP4': 'badge-indigo', 'AVI': 'badge-indigo', 'MOV': 'badge-indigo',
                'TXT': 'badge-gray', 'RTF': 'badge-gray',
            };

            return colorMap[ext] || 'badge-default';
        }
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            new FileInfoPanel();
        });
    } else {
        new FileInfoPanel();
    }

    // Export for external use
    window.FileInfoPanel = FileInfoPanel;

})();
(function () {
    'use strict';

    /**
     * Shared Control Bar Component
     * Handles select all and other controls for both table and card views
     */
    class SharedControlBar {
        constructor() {
            this.selectAllBtn = document.querySelector('#shared-select-all');
            this.currentView = this.getCurrentView();

            this.init();
        }

        init() {
            if (!this.selectAllBtn) return;

            this.setupEventListeners();
            this.updateSelectAllState();
        }

        setupEventListeners() {
            // Shared select all button
            this.selectAllBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.handleSelectAll();
            });

            // Listen for checkbox changes to update select all state
            document.addEventListener('change', (e) => {
                if (e.target.type === 'checkbox' &&
                    (e.target.classList.contains('card-checkbox') ||
                     e.target.name === 'batch[]')) {
                    this.updateSelectAllState();
                }
            });

            // Listen for view changes
            document.addEventListener('click', (e) => {
                if (e.target.closest('.view-toggle-btn')) {
                    setTimeout(() => {
                        this.currentView = this.getCurrentView();
                        this.updateSelectAllState();
                    }, 100);
                }
            });
        }

        getCurrentView() {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get('view') === 'cards' ? 'cards' : 'table';
        }

        getCheckboxes() {
            if (this.currentView === 'cards') {
                return document.querySelectorAll('.card-checkbox');
            } else {
                return document.querySelectorAll('input[name="batch[]"]');
            }
        }

        handleSelectAll() {
            const checkboxes = this.getCheckboxes();
            const selectedCount = Array.from(checkboxes).filter(cb => cb.checked).length;
            const totalCheckboxes = checkboxes.length;

            // Determine what action to take based on current state
            let shouldCheck;
            if (selectedCount === 0) {
                // Nothing selected - select all
                shouldCheck = true;
            } else if (selectedCount === totalCheckboxes) {
                // All selected - deselect all
                shouldCheck = false;
            } else {
                // Some selected - deselect all (clear selection)
                shouldCheck = false;
            }

            checkboxes.forEach(checkbox => {
                if (checkbox.checked !== shouldCheck) {
                    checkbox.checked = shouldCheck;

                    // Trigger change event for other components
                    const changeEvent = new Event('change', { bubbles: true });
                    checkbox.dispatchEvent(changeEvent);
                }
            });

            this.updateSelectAllState();
        }

        updateSelectAllState() {
            const checkboxes = this.getCheckboxes();
            const totalCheckboxes = checkboxes.length;
            const selectedCount = Array.from(checkboxes).filter(cb => cb.checked).length;

            if (!this.selectAllBtn) return;

            const icon = this.selectAllBtn.querySelector('i');
            const text = this.selectAllBtn.querySelector('span');

            if (selectedCount === 0) {
                // None selected
                this.selectAllBtn.classList.remove('active');
                if (icon) icon.className = 'fa fa-square-o';
                if (text) text.textContent = 'Select All';
            } else if (selectedCount === totalCheckboxes) {
                // All selected
                this.selectAllBtn.classList.add('active');
                if (icon) icon.className = 'fa fa-check-square-o';
                if (text) text.textContent = 'Deselect All';
            } else {
                // Some selected
                this.selectAllBtn.classList.add('active');
                if (icon) icon.className = 'fa fa-minus-square-o';
                if (text) text.textContent = `Deselect (${selectedCount})`;
            }
        }

        refresh() {
            this.currentView = this.getCurrentView();
            this.updateSelectAllState();
        }
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            new SharedControlBar();
        });
    } else {
        new SharedControlBar();
    }

    // Export for external use
    window.SharedControlBar = SharedControlBar;

})();
(function () {
    'use strict';

    /**
     * Theme Toggle Component
     * Handles switching between light and dark themes
     */
    class ThemeToggle {
        constructor() {
            this.storageKey = 'projectsend-theme';
            this.toggleButton = document.getElementById('theme-toggle');
            this.darkIcon = document.querySelector('.theme-icon-dark');
            this.lightIcon = document.querySelector('.theme-icon-light');

            this.init();
        }

        init() {
            // Set initial theme based on stored preference or system preference
            this.setInitialTheme();

            // Add event listener to toggle button
            if (this.toggleButton) {
                this.toggleButton.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.toggleTheme();
                });
            }

            // Listen for system theme changes
            if (window.matchMedia) {
                const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
                mediaQuery.addEventListener('change', () => {
                    if (!this.hasStoredPreference()) {
                        this.applyTheme(this.getSystemTheme());
                    }
                });
            }
        }

        setInitialTheme() {
            const storedTheme = this.getStoredTheme();
            const systemTheme = this.getSystemTheme();
            const theme = storedTheme || systemTheme;

            this.applyTheme(theme);
        }

        getStoredTheme() {
            try {
                return localStorage.getItem(this.storageKey);
            } catch (e) {
                return null;
            }
        }

        getSystemTheme() {
            // Always default to light mode, ignoring system preference
            // Uncomment the lines below to respect system preference
            // if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            //     return 'dark';
            // }
            return 'light';
        }

        hasStoredPreference() {
            return this.getStoredTheme() !== null;
        }

        getCurrentTheme() {
            return document.documentElement.getAttribute('data-theme') || 'light';
        }

        toggleTheme() {
            const currentTheme = this.getCurrentTheme();
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';

            this.applyTheme(newTheme);
            this.storeTheme(newTheme);
        }

        applyTheme(theme) {
            // Apply theme to document
            document.documentElement.setAttribute('data-theme', theme);

            // Update button icons
            this.updateIcons(theme);

            // Update button title
            this.updateButtonTitle(theme);

            // Dispatch theme change event
            this.dispatchThemeChangeEvent(theme);
        }

        updateIcons(theme) {
            if (!this.darkIcon || !this.lightIcon) return;

            if (theme === 'dark') {
                this.darkIcon.style.display = 'none';
                this.lightIcon.style.display = 'block';
            } else {
                this.darkIcon.style.display = 'block';
                this.lightIcon.style.display = 'none';
            }
        }

        updateButtonTitle(theme) {
            if (!this.toggleButton) return;

            const titleText = theme === 'dark'
                ? 'Switch to light theme'
                : 'Switch to dark theme';

            this.toggleButton.setAttribute('title', titleText);
        }

        storeTheme(theme) {
            try {
                localStorage.setItem(this.storageKey, theme);
            } catch (e) {
                console.warn('Failed to store theme preference:', e);
            }
        }

        dispatchThemeChangeEvent(theme) {
            const event = new CustomEvent('themechange', {
                detail: { theme: theme }
            });
            document.dispatchEvent(event);
        }
    }

    // Initialize theme toggle when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            new ThemeToggle();
        });
    } else {
        new ThemeToggle();
    }

    // Export for potential external usage
    window.ThemeToggle = ThemeToggle;

})();
(function () {
    'use strict';

    /**
     * View Toggle Component
     * Simple component that handles view toggle button states
     * Layout switching is now handled by URL parameters and server-side cookie persistence
     */
    class ViewToggle {
        constructor() {
            this.init();
        }

        init() {
            // View toggle is now handled by simple links
            // No JavaScript interaction needed - the server handles everything via URL params and cookies

            // Optional: Add any visual enhancements like hover effects
            this.setupHoverEffects();
        }

        setupHoverEffects() {
            const toggleButtons = document.querySelectorAll('.view-toggle-buttons a');
            toggleButtons.forEach(button => {
                button.addEventListener('mouseenter', () => {
                    if (!button.classList.contains('btn-primary')) {
                        button.style.transform = 'translateY(-1px)';
                    }
                });

                button.addEventListener('mouseleave', () => {
                    button.style.transform = '';
                });
            });
        }
    }

    // Initialize view toggle when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            new ViewToggle();
        });
    } else {
        new ViewToggle();
    }

    // Export for external use
    window.ViewToggle = ViewToggle;

})();
(function () {
    'use strict';

    $(document).ready(function() {
        admin.parts.main();
        admin.parts.misc();
        admin.parts.mainMenuSidebar();
        admin.parts.bulkActions();
        admin.parts.jqueryValidationCustomMethods();
        admin.parts.passwordVisibilityToggle();
        admin.parts.loadCKEditor();
        admin.parts.downloadCookieHandler();
        admin.parts.select2();
        admin.parts.publicLinksPopup();
        admin.parts.generatePassword();

        // Switch pages
        switch ($("body").data("page-id")) {
            case 'install':
                admin.pages.install();
                break;
            case 'login':
                admin.pages.loginForm();
                admin.pages.loginLdapForm();
                admin.parts.login2faInputs();
                admin.parts.loginTabs();
                break;
            case 'dashboard':
                admin.pages.dashboard();
                admin.parts.widgetStatistics();
                admin.parts.widgetActionLog();
                admin.parts.widgetNews();
                break;
            case 'categories_list':
                admin.pages.categoriesAdmin();
                break;
            case 'clients_memberships_requests':
                admin.pages.clientsAccountsRequests();
                break;
            case 'clients_accounts_requests':
                admin.pages.clientsAccountsRequests();
                break;
            case 'file_editor':
                admin.pages.fileEditor();
                break;
            case 'client_form':
                admin.pages.clientForm();
                break;
            case 'user_form':
                admin.pages.userForm();
                break;
            case 'group_form':
                admin.pages.groupForm();
                break;
            case 'email_templates':
                admin.pages.emailTemplates();
                break;
            case 'default_template':
            case 'manage_files':
                admin.parts.filePreviewModal();
                admin.parts.foldersAdmin();
                break;
            case 'manage_downloads':
                admin.parts.filePreviewModal();
                break;
            case 'reset_password_enter_email':
                admin.pages.resetPasswordEnterEmail();
                break;
            case 'reset_password_enter_new':
                admin.pages.resetPasswordEnterNew();
                break;
            case 'upload_form':
                admin.pages.uploadForm();
                break;
            case 'import_orphans':
                admin.pages.importOrphans();
                break;
            case 'options':
                admin.pages.options();
                break;
            case 'asset_editor':
                admin.pages.assetEditor();
                break;
            case 'public_files_list':
                admin.parts.filePreviewModal();
                admin.pages.publicFilesList();
                break;
            case 'public_download':
                admin.parts.filePreviewModal();
                break;
            case 'thumbnails_regenerate':
                admin.pages.thumbnailsRegenerate();
                break;
            case 'roles':
                admin.pages.roles();
                break;
            case 'role_permissions':
                admin.pages.rolePermissions();
                break;
            case 'updates':
                admin.pages.updates();
                break;
            case 'import_external':
                admin.pages.import_external();
                break;
            case 'integration_form':
                admin.pages.integration_form();
                break;
            default:
                // do nothing
                break;
        }
    });
})();
function htmlEncode(str)
{
    return String(str).replace(/[^\w. ]/gi, function(c){
        return '&#'+c.charCodeAt(0)+';';
    });
}

// Adapted from https://stackoverflow.com/questions/400212/how-do-i-copy-to-the-clipboard-in-javascript
function fallbackCopyTextToClipboard(text) {
    var textArea = document.createElement("textarea");
    textArea.value = text;
    
    // Avoid scrolling to bottom
    textArea.style.top = "0";
    textArea.style.left = "0";
    textArea.style.position = "fixed";

    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();

    try {
        var successful = document.execCommand('copy');
        if (successful) {
            toastr.success(json_strings.translations.copy_ok)
        } else {
            toastr.error(json_strings.translations.copy_error)    
        }
    } catch (err) {
        toastr.error(json_strings.translations.copy_error)
    }

    document.body.removeChild(textArea);
}

function copyTextToClipboard(text) {
    if (!navigator.clipboard) {
        fallbackCopyTextToClipboard(text);
        return;
    }

    navigator.clipboard.writeText(text).then(function() {
        toastr.success(json_strings.translations.copy_ok)
    }, function(err) {
        toastr.error(json_strings.translations.copy_ok)
    });
}

//https://gist.github.com/fnicollier/4258461
function insertAtCaret(areaId,text) {
    var txtarea = document.getElementById(areaId);
    var scrollPos = txtarea.scrollTop;
    var strPos = 0;
    var br = ((txtarea.selectionStart || txtarea.selectionStart == '0') ? 
        "ff" : (document.selection ? "ie" : false ) );
    if (br == "ie") { 
        txtarea.focus();
        var range = document.selection.createRange();
        range.moveStart ('character', -txtarea.value.length);
        strPos = range.text.length;
    }
    else if (br == "ff") strPos = txtarea.selectionStart;

    var front = (txtarea.value).substring(0,strPos);  
    var back = (txtarea.value).substring(strPos,txtarea.value.length); 
    txtarea.value=front+text+back;
    strPos = strPos + text.length;
    if (br == "ie") { 
        txtarea.focus();
        var range = document.selection.createRange();
        range.moveStart ('character', -txtarea.value.length);
        range.moveStart ('character', strPos);
        range.moveEnd ('character', 0);
        range.select();
    }
    else if (br == "ff") {
        txtarea.selectionStart = strPos;
        txtarea.selectionEnd = strPos;
        txtarea.focus();
    }
    txtarea.scrollTop = scrollPos;
}

function urlParamExists(param, value = null)
{
    var urlParams = new URLSearchParams(window.location.search);
    var search = urlParams.get(param);
    
    if(search) {
        return true;
    }

    if (value != null) {
        if(search == value) {
            return true;
        }
    }
    
    return false;
}

function isNumeric(value){
    if (typeof value != "string") {
        return false;
    }

    return !isNaN(value) && !isNaN(parseFloat(value))
}

function replaceAll(str,mapObj){
    var re = new RegExp(Object.keys(mapObj).join("|"),"gi");

    return str.replace(re, function(matched){
        return mapObj[matched.toLowerCase()];
    });
}

function elementExists(element) {
    return (element != null && element != 'undefined');
}

(function () {
    'use strict';

    admin.pages.assetEditor = function () {
        $(document).ready(function()
        {
            const types_allowed = ['js', 'css', 'html'];
            const type = $('#asset_language').val();
            var mode;
            if (types_allowed.includes(type)) {
                switch (type) {
                    case 'css': mode = 'css'; break;
                    case 'js': mode = 'javascript'; break;
                    case 'html': mode = 'htmlmixed'; break;
                }
            }
            // console.log(mode);
            var editor = CodeMirror.fromTextArea(document.getElementById("content"), {
                lineNumbers: true,
                mode: mode,
                //theme: 'neo',
                lineWrapping: true
            });
        });
    };
})();


(function () {
    'use strict';

    admin.pages.categoriesAdmin = function () {

        $(document).ready(function(){
            var validator = $("#process_category").validate({
                rules: {
                    category_name: {
                        required: true,
                    }
                },
                messages: {
                    category_name: {
                        required: json_strings.validation.no_name,
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                },
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.clientForm = function () {
        var form = document.getElementById('client_form');
        var formChanged = false;
        var formSubmitting = false;
        var initialFormData = form ? new FormData(form) : null;

        $(document).ready(function(){
            var form_type = $("#client_form").data('form-type');

            var validator = $("#client_form").validate({
                rules: {
                    name: {
                        required: true
                    },
                    username: {
                        required: true,
                        minlength: json_strings.character_limits.user_min,
                        maxlength: json_strings.character_limits.user_max,
                        alphanumericUsername: true
                    },
                    email: {
                        required: true,
                        email: true
                    },
                    max_file_size: {
                        required: {
                            param: true,
                            depends: function(element) {
                                return form_type != 'new_client_self';
                            }
                        },
                        digits: true
                    },
                    password: {
                        required: {
                            param: true,
                            depends: function(element) {
                                if (form_type == 'new_client' || form_type == 'new_client_self') {
                                    return true;
                                }
                                if (form_type == 'edit_client' || form_type == 'edit_client_self') {
                                    if ($.trim($("#password").val()).length > 0) {
                                        return true;
                                    }
                                }
                                return false;
                            }
                        },
                        minlength: json_strings.character_limits.password_min,
                        maxlength: json_strings.character_limits.password_max,
                        passwordValidCharacters: true
                    }
                },
                messages: {
                    name: {
                        required: json_strings.validation.no_name
                    },
                    username: {
                        required: json_strings.validation.no_user,
                        minlength: json_strings.validation.length_user,
                        maxlength: json_strings.validation.length_user,
                        alphanumericUsername: json_strings.validation.alpha_user
                    },
                    email: {
                        required: json_strings.validation.no_email,
                        email: json_strings.validation.invalid_email
                    },
                    max_file_size: {
                        digits: json_strings.validation.file_size
                    },
                    password: {
                        required: json_strings.validation.no_pass,
                        minlength: json_strings.validation.length_pass,
                        maxlength: json_strings.validation.length_pass,
                        passwordValidCharacters: json_strings.validation.alpha_pass
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });
        });

        // Track form changes for unsaved changes warning
        if (form) {
            // Track changes to all form inputs
            form.addEventListener('change', function() {
                formChanged = true;
            });

            // Also track text input changes
            var textInputs = form.querySelectorAll('input[type="text"], input[type="email"], input[type="password"], input[type="number"], input[type="checkbox"], textarea, select');
            textInputs.forEach(function(input) {
                input.addEventListener('input', function() {
                    // Check if form has actually changed from initial state
                    var currentFormData = new FormData(form);
                    formChanged = !areFormDataEqual(initialFormData, currentFormData);
                });
            });

            // Set flag when form is being submitted
            form.addEventListener('submit', function() {
                formSubmitting = true;
            });

            // Helper function to compare FormData objects
            function areFormDataEqual(formData1, formData2) {
                if (!formData1 || !formData2) return false;

                var entries1 = Array.from(formData1.entries());
                var entries2 = Array.from(formData2.entries());

                if (entries1.length !== entries2.length) {
                    return false;
                }

                for (var i = 0; i < entries1.length; i++) {
                    if (entries1[i][0] !== entries2[i][0] || entries1[i][1] !== entries2[i][1]) {
                        return false;
                    }
                }

                return true;
            }

            // Warn user before leaving if there are unsaved changes
            window.addEventListener('beforeunload', function(e) {
                if (formChanged && !formSubmitting) {
                    var confirmationMessage = 'You have unsaved changes. Are you sure you want to leave this page?';
                    e.returnValue = confirmationMessage;
                    return confirmationMessage;
                }
            });

            // Handle navigation away from the page (for links within the application)
            document.addEventListener('click', function(e) {
                // Check if it's a link that would navigate away
                var target = e.target.closest('a');
                if (target && target.href && !target.href.startsWith('#') && !target.classList.contains('delete-confirm')) {
                    if (formChanged && !formSubmitting) {
                        e.preventDefault();
                        var targetUrl = target.href;

                        Swal.fire({
                            title: 'Unsaved Changes',
                            text: 'You have unsaved changes. Are you sure you want to leave this page?',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#d33',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Yes, leave page',
                            cancelButtonText: 'Stay on page',
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        }).then(function(result) {
                            if (result.isConfirmed) {
                                formSubmitting = true; // Prevent the beforeunload warning
                                window.location.href = targetUrl;
                            }
                        });
                    }
                }
            });
        }
    };
})();
(function () {
    'use strict';

    admin.pages.clientsAccountsRequests = function () {

        $(document).ready(function(){
            $('.change_all').click(function(e) {
                e.preventDefault();
                var target = $(this).data('target');
                var check = $(this).data('check');
                $("input[data-client='"+target+"']").prop("checked",check).change();
                check_client(target);
            });
            
            $('.account_action').on("change", function() {
                if ( $(this).prop('checked') == false )  {
                    var target = $(this).data('client');
                    $(".membership_action[data-client='"+target+"']").prop("checked",false).change();
                }
            });
    
            $('.checkbox_toggle').change(function() {
                var target = $(this).data('client');
                check_client(target);
            });
    
            function check_client(client_id) {
                $("input[data-clientid='"+client_id+"']").prop("checked",true);
            }
        });
    };
})();
(function () {
    'use strict';

    admin.pages.customFieldForm = function () {
        document.addEventListener('DOMContentLoaded', function() {
            const fieldTypeSelect = document.getElementById('field_type');
            const fieldOptionsContainer = document.getElementById('field_options_container');
            const fieldOptionsTextarea = document.getElementById('field_options');
            const fieldNameInput = document.getElementById('field_name');
            const fieldLabelInput = document.getElementById('field_label');

            // Show/hide field options based on field type
            function toggleFieldOptions() {
                if (fieldTypeSelect.value === 'select') {
                    fieldOptionsContainer.style.display = 'block';
                    fieldOptionsTextarea.setAttribute('required', 'required');
                } else {
                    fieldOptionsContainer.style.display = 'none';
                    fieldOptionsTextarea.removeAttribute('required');
                }
            }

            // Auto-generate field name from label
            function generateFieldName() {
                const label = fieldLabelInput.value.toLowerCase();
                const fieldName = label
                    .replace(/[^\w\s]/g, '') // Remove special characters
                    .replace(/\s+/g, '_') // Replace spaces with underscores
                    .substring(0, 50); // Limit length

                if (fieldName && !fieldNameInput.value) {
                    fieldNameInput.value = fieldName;
                }
            }

            // Validate field name format
            function validateFieldName() {
                const fieldName = fieldNameInput.value;
                const validPattern = /^[a-z0-9_]+$/;

                if (fieldName && !validPattern.test(fieldName)) {
                    fieldNameInput.setCustomValidity('Field name can only contain lowercase letters, numbers, and underscores.');
                } else {
                    fieldNameInput.setCustomValidity('');
                }
            }

            // Event listeners
            if (fieldTypeSelect) {
                fieldTypeSelect.addEventListener('change', toggleFieldOptions);
                // Initialize on page load
                toggleFieldOptions();
            }

            if (fieldLabelInput) {
                fieldLabelInput.addEventListener('blur', generateFieldName);
            }

            if (fieldNameInput) {
                fieldNameInput.addEventListener('input', validateFieldName);
                // Convert to lowercase and replace spaces with underscores as user types
                fieldNameInput.addEventListener('input', function() {
                    const cursorPosition = this.selectionStart;
                    const originalValue = this.value;
                    const newValue = originalValue.toLowerCase().replace(/\s+/g, '_');

                    if (originalValue !== newValue) {
                        this.value = newValue;
                        this.setSelectionRange(cursorPosition, cursorPosition);
                    }
                });
            }
        });
    };
})();
(function () {
    'use strict';

    admin.pages.custom_fields = function () {
        var draggedElement = null;
        var draggedOver = null;

        function initCustomFields() {
            // Handle delete confirmations
            var deleteButtons = document.querySelectorAll('.delete-confirm');

            deleteButtons.forEach(function(button) {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    var deleteUrl = this.getAttribute('href');

                    Swal.fire({
                        title: 'Are you sure?',
                        text: "This will permanently delete this custom field and all its data. This action cannot be undone!",
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonColor: '#d33',
                        cancelButtonColor: '#6c757d',
                        confirmButtonText: 'Yes, delete it!',
                        cancelButtonText: 'Cancel',
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    }).then(function(result) {
                        if (result.isConfirmed) {
                            window.location.href = deleteUrl;
                        }
                    });
                });
            });

            // Initialize drag and drop functionality
            initDragAndDrop();
        }

        function initDragAndDrop() {
            var tbody = document.querySelector('#custom_fields_tbl tbody');
            if (!tbody) return;

            var rows = tbody.querySelectorAll('tr');

            rows.forEach(function(row, index) {
                // Make the entire row draggable but only when dragging from the handle
                var dragHandle = row.querySelector('.drag-handle');
                if (!dragHandle) {
                    return;
                }

                // Set up drag handle mouse events
                dragHandle.addEventListener('mousedown', function(e) {
                    row.draggable = true;
                });

                // Make sure row is draggable
                row.draggable = true;

                row.addEventListener('dragstart', function(e) {
                    draggedElement = this;
                    this.classList.add('dragging');
                    e.dataTransfer.effectAllowed = 'move';
                    e.dataTransfer.setData('text/html', this.outerHTML);
                });

                row.addEventListener('dragend', function(e) {
                    this.classList.remove('dragging');
                    row.draggable = false; // Disable dragging when not actively dragging

                    // Remove all dragover effects
                    var allRows = tbody.querySelectorAll('tr');
                    allRows.forEach(function(r) {
                        r.classList.remove('drag-over');
                    });

                    draggedElement = null;
                    draggedOver = null;
                });

                row.addEventListener('dragover', function(e) {
                    e.preventDefault();
                    e.dataTransfer.dropEffect = 'move';

                    if (draggedElement && draggedElement !== this) {
                        // Remove drag-over from all rows first
                        var allRows = tbody.querySelectorAll('tr');
                        allRows.forEach(function(r) {
                            r.classList.remove('drag-over');
                        });

                        // Add to current row
                        this.classList.add('drag-over');
                        draggedOver = this;
                    }
                });

                row.addEventListener('dragleave', function(e) {
                    // Only remove if we're actually leaving the row (not entering a child element)
                    var rect = this.getBoundingClientRect();
                    var x = e.clientX;
                    var y = e.clientY;

                    if (x < rect.left || x > rect.right || y < rect.top || y > rect.bottom) {
                        this.classList.remove('drag-over');
                    }
                });

                row.addEventListener('drop', function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    if (draggedElement && draggedElement !== this) {
                        var rect = this.getBoundingClientRect();
                        var insertAfter;

                        // Check if this is the first row in the table
                        var isFirstRow = this.previousElementSibling === null;

                        if (isFirstRow) {
                            // For the first row, use a smaller threshold (top quarter)
                            // This makes it easier to place items at the very beginning
                            insertAfter = e.clientY > rect.top + rect.height / 4;
                        } else {
                            // For other rows, use the middle as threshold
                            insertAfter = e.clientY > rect.top + rect.height / 2;
                        }

                        // Move the dragged element in the DOM
                        if (insertAfter) {
                            this.parentNode.insertBefore(draggedElement, this.nextSibling);
                        } else {
                            this.parentNode.insertBefore(draggedElement, this);
                        }

                        // Update sort order on server
                        updateSortOrder();
                    }

                    // Clean up
                    this.classList.remove('drag-over');
                    var allRows = tbody.querySelectorAll('tr');
                    allRows.forEach(function(r) {
                        r.classList.remove('drag-over');
                    });
                });
            });
        }

        function updateSortOrder() {
            var tbody = document.querySelector('#custom_fields_tbl tbody');
            if (!tbody) return;

            var rows = tbody.querySelectorAll('tr');
            var sortData = [];

            rows.forEach(function(row, index) {
                // Try to get field ID from the row's data attribute first
                var fieldId = parseInt(row.getAttribute('data-field-id'));

                // If not found on row, try the first cell
                if (isNaN(fieldId)) {
                    var firstCell = row.querySelector('.drag-handle-cell');
                    if (firstCell) {
                        fieldId = parseInt(firstCell.getAttribute('data-field-id'));
                    }
                }

                // Last fallback: try the hidden span
                if (isNaN(fieldId)) {
                    var hiddenSpan = row.querySelector('.field-id-hidden');
                    if (hiddenSpan) {
                        fieldId = parseInt(hiddenSpan.textContent);
                    }
                }

                // Only add if we have a valid field ID
                if (!isNaN(fieldId) && fieldId > 0) {
                    sortData.push({
                        id: fieldId,
                        sort_order: index + 1
                    });
                }
            });

            // Show loading indicator
            toastr.info('Updating field order...', '', {
                timeOut: 0,
                extendedTimeOut: 0,
                closeButton: false,
                tapToDismiss: false
            });

            // Send AJAX request to update sort order
            fetch(json_strings.uri.base + 'process.php?do=update_custom_fields_order', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify({
                    csrf_token: document.getElementById('csrf_token').value,
                    sort_data: sortData
                })
            })
            .then(function(response) {
                return response.json();
            })
            .then(function(data) {
                toastr.clear();

                if (data.status === 'success') {
                    toastr.success(data.message);
                } else {
                    toastr.error(data.message || 'Failed to update order');

                    // Reload page to restore original order
                    setTimeout(function() {
                        window.location.reload();
                    }, 2000);
                }
            })
            .catch(function(error) {
                toastr.clear();

                toastr.error('Failed to update order. Please try again.');

                // Reload page to restore original order
                setTimeout(function() {
                    window.location.reload();
                }, 2000);
            });
        }

        // Check if DOM is already loaded or wait for it
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initCustomFields);
        } else {
            initCustomFields();
        }
    };
})();
(function () {
    'use strict';

    admin.pages.custom_fields_form = function () {
        function initCustomFieldsForm() {
            var fieldLabel = document.getElementById('field_label');
            var fieldName = document.getElementById('field_name');
            var fieldType = document.getElementById('field_type');
            var fieldOptionsContainer = document.getElementById('field_options_container');
            var form = document.getElementById('custom_field_form');
            var formChanged = false;
            var formSubmitting = false;
            var initialFormData = new FormData(form);

            // Function to create slug from text
            function createSlug(text) {
                return text.toString()
                    .toLowerCase()
                    .trim()
                    .replace(/[\s\-]+/g, '_')          // Replace spaces and hyphens with underscores
                    .replace(/[^\w_]+/g, '')           // Remove all non-word chars except underscores
                    .replace(/\_\_+/g, '_')            // Replace multiple underscores with single underscore
                    .replace(/^_+/, '')                // Trim underscores from start
                    .replace(/_+$/, '');               // Trim underscores from end
            }

            // Auto-generate field_name from field_label on blur (only for add form)
            if (fieldLabel && fieldName && !fieldName.hasAttribute('readonly')) {
                var autoGenerate = true;

                // Function to generate slug
                function generateSlug() {
                    if (autoGenerate && fieldName.value.trim() === '') {
                        var slug = createSlug(fieldLabel.value);
                        fieldName.value = slug;
                    }
                }

                // Generate on blur
                fieldLabel.addEventListener('blur', generateSlug);

                // Also generate on real-time typing if field_name is empty
                //fieldLabel.addEventListener('input', generateSlug);

                // Stop auto-generating once user manually edits field_name
                // But re-enable if field_name is cleared
                fieldName.addEventListener('input', function() {
                    if (fieldName.value.trim() === '') {
                        autoGenerate = true;
                    } else {
                        autoGenerate = false;
                    }
                });
            }

            // Show/hide field options based on field type
            if (fieldType && fieldOptionsContainer) {
                function toggleFieldOptions() {
                    var fieldOptionsHelp = document.getElementById('field_options_help');

                    if (fieldType.value === 'select' || fieldType.value === 'checkbox') {
                        fieldOptionsContainer.style.display = 'block';
                        // Make field_options required when visible (optional for checkbox)
                        var fieldOptions = document.getElementById('field_options');
                        if (fieldOptions) {
                            if (fieldType.value === 'select') {
                                fieldOptions.setAttribute('required', 'required');
                                fieldOptions.setAttribute('rows', '5');
                            } else {
                                // Checkbox label is optional (defaults to "Yes")
                                fieldOptions.removeAttribute('required');
                                fieldOptions.setAttribute('rows', '2');
                            }
                        }

                        // Update help text based on type
                        if (fieldOptionsHelp) {
                            if (fieldType.value === 'select') {
                                fieldOptionsHelp.innerHTML = 'For select fields: Enter one option per line.';
                            } else {
                                fieldOptionsHelp.innerHTML = 'For checkbox fields: Enter the checkbox label (e.g., "I agree to the terms"). Leave empty for default "Yes".';
                            }
                        }
                    } else {
                        fieldOptionsContainer.style.display = 'none';
                        // Remove required when hidden
                        var fieldOptions = document.getElementById('field_options');
                        if (fieldOptions) {
                            fieldOptions.removeAttribute('required');
                        }
                    }
                }

                // Initial check on page load
                toggleFieldOptions();

                // Toggle on field type change
                fieldType.addEventListener('change', toggleFieldOptions);
            }

            // Handle delete confirmation for the delete button in edit form
            var deleteButtons = document.querySelectorAll('.delete-confirm');
            deleteButtons.forEach(function(button) {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    var deleteUrl = this.getAttribute('href');

                    Swal.fire({
                        title: 'Are you sure?',
                        text: "This will permanently delete this custom field and all its data. This action cannot be undone!",
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonColor: '#d33',
                        cancelButtonColor: '#6c757d',
                        confirmButtonText: 'Yes, delete it!',
                        cancelButtonText: 'Cancel',
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    }).then(function(result) {
                        if (result.isConfirmed) {
                            window.location.href = deleteUrl;
                        }
                    });
                });
            });

            // Track form changes
            if (form) {
                // Track changes to all form inputs
                form.addEventListener('change', function() {
                    formChanged = true;
                });

                // Also track text input changes
                var textInputs = form.querySelectorAll('input[type="text"], input[type="checkbox"], textarea, select');
                textInputs.forEach(function(input) {
                    input.addEventListener('input', function() {
                        // Check if form has actually changed from initial state
                        var currentFormData = new FormData(form);
                        formChanged = !areFormDataEqual(initialFormData, currentFormData);
                    });
                });

                // Set flag when form is being submitted
                form.addEventListener('submit', function() {
                    formSubmitting = true;
                });

                // Helper function to compare FormData objects
                function areFormDataEqual(formData1, formData2) {
                    var entries1 = Array.from(formData1.entries());
                    var entries2 = Array.from(formData2.entries());

                    if (entries1.length !== entries2.length) {
                        return false;
                    }

                    for (var i = 0; i < entries1.length; i++) {
                        if (entries1[i][0] !== entries2[i][0] || entries1[i][1] !== entries2[i][1]) {
                            return false;
                        }
                    }

                    return true;
                }

                // Warn user before leaving if there are unsaved changes
                window.addEventListener('beforeunload', function(e) {
                    if (formChanged && !formSubmitting) {
                        var confirmationMessage = 'You have unsaved changes. Are you sure you want to leave this page?';
                        e.returnValue = confirmationMessage;
                        return confirmationMessage;
                    }
                });

                // Handle navigation away from the page (for links within the application)
                document.addEventListener('click', function(e) {
                    // Check if it's a link that would navigate away
                    var target = e.target.closest('a');
                    if (target && target.href && !target.href.startsWith('#') && !target.classList.contains('delete-confirm')) {
                        if (formChanged && !formSubmitting) {
                            e.preventDefault();
                            var targetUrl = target.href;

                            Swal.fire({
                                title: 'Unsaved Changes',
                                text: 'You have unsaved changes. Are you sure you want to leave this page?',
                                icon: 'warning',
                                showCancelButton: true,
                                confirmButtonColor: '#d33',
                                cancelButtonColor: '#6c757d',
                                confirmButtonText: 'Yes, leave page',
                                cancelButtonText: 'Stay on page',
                                showClass: {
                                    popup: 'animate__animated animate__fadeIn'
                                },
                                hideClass: {
                                    popup: 'animate__animated animate__fadeOut'
                                }
                            }).then(function(result) {
                                if (result.isConfirmed) {
                                    formSubmitting = true; // Prevent the beforeunload warning
                                    window.location.href = targetUrl;
                                }
                            });
                        }
                    }
                });
            }
        }

        // Check if DOM is already loaded or wait for it
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initCustomFieldsForm);
        } else {
            initCustomFieldsForm();
        }
    };
})();
(function () {
    'use strict';

    admin.pages.dashboard = function () {

        $(document).ready(function(){
            initCollapsibleWidgets();
            initDraggableWidgets();
        });

        function initCollapsibleWidgets() {
            // Convert h4 headers to proper widget headers with controls
            $('.widget h4').each(function() {
                const $h4 = $(this);
                const $widget = $h4.closest('.widget');
                const $container = $widget.closest('.widget-container');
                const widgetId = $widget.attr('id');

                if (!widgetId) return; // Skip widgets without IDs

                const headerTitle = $h4.text();

                // Create new header structure
                const headerHtml = `
                    <div class="widget-header">
                        <div class="widget-title">
                            <span class="widget-title-text">${headerTitle}</span>
                        </div>
                        <div class="widget-controls">
                            <button class="widget-control-btn widget-narrower-btn" title="Make Narrower" data-action="narrower">
                                <i class="fa fa-minus"></i>
                            </button>
                            <button class="widget-control-btn widget-wider-btn" title="Make Wider" data-action="wider">
                                <i class="fa fa-plus"></i>
                            </button>
                            <button class="widget-control-btn widget-collapse-btn" title="Collapse/Expand" data-action="collapse">
                                <i class="fa fa-chevron-up"></i>
                            </button>
                        </div>
                    </div>
                `;

                // Replace h4 with new header
                $h4.replaceWith(headerHtml);

                // Check saved state from cookie
                const isCollapsed = Cookies.get('widget_' + widgetId + '_collapsed') === 'true';
                if (isCollapsed) {
                    collapseWidget($widget, false); // false = no animation on init
                }

                // Check saved width from cookie
                const savedWidth = Cookies.get('widget_' + widgetId + '_width');
                if (savedWidth) {
                    setWidgetWidth($container, parseInt(savedWidth));
                }
            });

            // Handle control button clicks
            $(document).on('click', '.widget-control-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();

                const action = $(this).data('action');
                const $widget = $(this).closest('.widget');
                const $container = $widget.closest('.widget-container');
                const widgetId = $widget.attr('id');

                if (action === 'collapse') {
                    const $content = $widget.find('.widget_int');

                    if ($content.is(':visible')) {
                        collapseWidget($widget, true);
                        Cookies.set('widget_' + widgetId + '_collapsed', 'true', { expires: 365 });
                    } else {
                        expandWidget($widget, true);
                        Cookies.set('widget_' + widgetId + '_collapsed', 'false', { expires: 365 });
                    }
                } else if (action === 'wider') {
                    makeWidgetWider($container, widgetId);
                } else if (action === 'narrower') {
                    makeWidgetNarrower($container, widgetId);
                }
            });

            // Handle title area clicks for collapse (not on control buttons)
            $(document).on('click', '.widget-title', function(e) {
                e.preventDefault();
                const $widget = $(this).closest('.widget');
                const $content = $widget.find('.widget_int');
                const widgetId = $widget.attr('id');

                if ($content.is(':visible')) {
                    collapseWidget($widget, true);
                    Cookies.set('widget_' + widgetId + '_collapsed', 'true', { expires: 365 });
                } else {
                    expandWidget($widget, true);
                    Cookies.set('widget_' + widgetId + '_collapsed', 'false', { expires: 365 });
                }
            });
        }

        function collapseWidget($widget, animate = true) {
            const $content = $widget.find('.widget_int');
            const $collapseBtn = $widget.find('.widget-collapse-btn i');

            if (animate) {
                $content.slideUp(200);
            } else {
                $content.hide();
            }

            $collapseBtn.removeClass('fa-chevron-up').addClass('fa-chevron-down');
            $widget.addClass('widget-collapsed');
        }

        function expandWidget($widget, animate = true) {
            const $content = $widget.find('.widget_int');
            const $collapseBtn = $widget.find('.widget-collapse-btn i');

            if (animate) {
                $content.slideDown(200);
            } else {
                $content.show();
            }

            $collapseBtn.removeClass('fa-chevron-down').addClass('fa-chevron-up');
            $widget.removeClass('widget-collapsed');
        }

        function makeWidgetWider($container, widgetId) {
            const currentWidth = getWidgetWidth($container);
            const maxColumns = getMaxColumns();

            if (currentWidth < maxColumns) {
                const newWidth = currentWidth + 1;
                setWidgetWidth($container, newWidth);
                Cookies.set('widget_' + widgetId + '_width', newWidth, { expires: 365 });
            }
        }

        function makeWidgetNarrower($container, widgetId) {
            const currentWidth = getWidgetWidth($container);

            if (currentWidth > 1) {
                const newWidth = currentWidth - 1;
                setWidgetWidth($container, newWidth);
                Cookies.set('widget_' + widgetId + '_width', newWidth, { expires: 365 });
            }
        }

        function getWidgetWidth($container) {
            // Check CSS grid-column-end style or data attribute
            for (let i = 1; i <= 4; i++) {
                if ($container.hasClass(`widget-width-${i}`)) {
                    return i;
                }
            }
            return 1; // default width
        }

        function setWidgetWidth($container, width) {
            // Remove existing width classes
            $container.removeClass('widget-width-1 widget-width-2 widget-width-3 widget-width-4');
            // Add new width class
            $container.addClass(`widget-width-${width}`);
        }

        function getMaxColumns() {
            // Determine max columns based on screen size
            const screenWidth = $(window).width();
            if (screenWidth < 768) return 1;  // Mobile
            if (screenWidth < 1200) return 2; // Tablet
            if (screenWidth < 1600) return 3; // Desktop
            return 4; // Large desktop
        }

        function initDraggableWidgets() {
            const container = document.getElementById('dashboard-widgets');
            if (!container) return;

            // Load saved widget order from cookies
            loadWidgetOrder();

            // Make widgets draggable
            $('.widget-container').each(function() {
                const widget = this;
                widget.draggable = true;

                // Drag start
                widget.addEventListener('dragstart', function(e) {
                    $(this).addClass('dragging');
                    e.dataTransfer.effectAllowed = 'move';
                    e.dataTransfer.setData('text/html', this.outerHTML);
                    e.dataTransfer.setData('text/plain', $(this).data('widget'));
                });

                // Drag end
                widget.addEventListener('dragend', function(e) {
                    $(this).removeClass('dragging');
                    $('.widget-container').removeClass('drag-over');
                });

                // Drag over
                widget.addEventListener('dragover', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.dataTransfer.dropEffect = 'move';

                    // Only show drag-over state if this isn't the dragged widget
                    if (!$(this).hasClass('dragging')) {
                        $('.widget-container').removeClass('drag-over'); // Remove from all others
                        $(this).addClass('drag-over');
                    }
                });

                // Drag leave
                widget.addEventListener('dragleave', function(e) {
                    // Only remove drag-over if we're actually leaving the widget
                    if (!this.contains(e.relatedTarget)) {
                        $(this).removeClass('drag-over');
                    }
                });

                // Drop
                widget.addEventListener('drop', function(e) {
                    e.preventDefault();
                    $(this).removeClass('drag-over');

                    const draggedWidget = $('.widget-container.dragging')[0];
                    if (draggedWidget && draggedWidget !== this) {
                        // Always insert the dragged widget before the drop target
                        this.parentNode.insertBefore(draggedWidget, this);

                        // Save new order
                        saveWidgetOrder();
                    }
                });
            });

            // Container drop handling
            container.addEventListener('dragover', function(e) {
                e.preventDefault();
                e.dataTransfer.dropEffect = 'move';
            });

            container.addEventListener('drop', function(e) {
                e.preventDefault();

                // Only handle if dropped on the container itself (not on a widget)
                if (e.target === this) {
                    const draggedWidget = $('.widget-container.dragging')[0];
                    if (draggedWidget) {
                        // Append to end if dropped on empty space
                        this.appendChild(draggedWidget);
                        saveWidgetOrder();
                    }
                }
            });
        }

        function saveWidgetOrder() {
            const widgetOrder = [];
            $('.widget-container').each(function() {
                const widgetId = $(this).data('widget');
                if (widgetId) {
                    widgetOrder.push(widgetId);
                }
            });

            // Save to cookie for 1 year
            Cookies.set('dashboard_widget_order', JSON.stringify(widgetOrder), { expires: 365 });
        }

        function loadWidgetOrder() {
            const savedOrder = Cookies.get('dashboard_widget_order');
            if (!savedOrder) return;

            try {
                const widgetOrder = JSON.parse(savedOrder);
                const container = $('#dashboard-widgets');

                // Reorder widgets based on saved order
                widgetOrder.forEach(function(widgetId) {
                    const widget = container.find(`[data-widget="${widgetId}"]`);
                    if (widget.length) {
                        container.append(widget);
                    }
                });
            } catch (e) {
                console.warn('Could not load widget order from cookies:', e);
            }
        }
    };
})();
(function () {
    'use strict';

    admin.pages.emailTemplates = function () {

        $(document).ready(function(){
            $(document).on('click', '.load_default', function(e) {
                e.preventDefault();

                var file = jQuery(this).data('file');
                var textarea = document.getElementById(jQuery(this).data('textarea'));
                var accept = confirm(json_strings.translations.email_templates.confirm_replace);
                
                if ( accept ) {
                    $.ajax({
                        url: json_strings.uri.base + "emails/"+file,
                        cache: false,
                        success: function (data){
                            textarea.value = data;
                        },
                        error: function() {
                            alert(json_strings.translations.email_templates.loading_error);
                        }
                    });
                }
            });
    
            $('.preview').click(function(e) {
                e.preventDefault();
                var type = jQuery(this).data('preview');
                var url = json_strings.uri.base+ 'email-preview.php?t=' + type;
                window.open(url, "previewWindow", "width=800,height=600,scrollbars=yes");
            });
        });

        $('.insert_tag').on('click', function(e) {
            var target = jQuery(this).data('target');
            var tag = $(this).data('tag');

            // Check if we have a CodeMirror editor for this textarea
            if (editors[target]) {
                var editor = editors[target];
                var doc = editor.getDoc();
                var cursor = doc.getCursor();
                doc.replaceRange(tag, cursor);
                editor.focus();
            } else {
                // Fallback to original function for regular textareas
                insertAtCaret(target, tag);
            }
        });

        // Template Gallery functionality
        $(document).on('click', '.btn-template-preview', function(e) {
            e.preventDefault();
            var templateId = $(this).data('template-id');
            showTemplatePreview(templateId);
        });

        // Template Apply functionality
        $(document).on('click', '.btn-template-apply', function(e) {
            e.preventDefault();
            var templateId = $(this).data('template-id');
            var templateName = $(this).data('template-name');
            showTemplateApplyConfirmation(templateId, templateName);
        });

        function showTemplatePreview(templateId) {
            // Create modal if it doesn't exist
            if ($('.template-preview-modal').length === 0) {
                var modalHtml = `
                    <div class="template-preview-modal">
                        <div class="template-preview-content">
                            <div class="template-preview-header">
                                <h3 class="template-preview-title">Template Preview</h3>
                                <button class="template-preview-close">
                                    <i class="fa fa-times"></i>
                                </button>
                            </div>
                            <div class="template-preview-body">
                                <iframe class="template-preview-iframe" src=""></iframe>
                            </div>
                        </div>
                    </div>
                `;
                $('body').append(modalHtml);
            }

            // Set iframe source
            var previewUrl = json_strings.uri.base + 'email-template-preview.php?template=' + templateId;
            $('.template-preview-iframe').attr('src', previewUrl);

            // Show modal
            $('.template-preview-modal').addClass('active');
        }

        // Close template preview modal
        $(document).on('click', '.template-preview-close', function(e) {
            e.preventDefault();
            $('.template-preview-modal').removeClass('active');
            // Clear iframe to stop any animations
            setTimeout(function() {
                $('.template-preview-iframe').attr('src', '');
            }, 300);
        });

        // Close modal when clicking outside content
        $(document).on('click', '.template-preview-modal', function(e) {
            if (e.target === this) {
                $('.template-preview-modal').removeClass('active');
                // Clear iframe to stop any animations
                setTimeout(function() {
                    $('.template-preview-iframe').attr('src', '');
                }, 300);
            }
        });

        // Prevent modal from closing when clicking inside content
        $(document).on('click', '.template-preview-content', function(e) {
            e.stopPropagation();
        });

        // Close modal with escape key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && $('.template-preview-modal').hasClass('active')) {
                $('.template-preview-modal').removeClass('active');
                setTimeout(function() {
                    $('.template-preview-iframe').attr('src', '');
                }, 300);
            }
        });

        function showTemplateApplyConfirmation(templateId, templateName) {
            Swal.fire({
                title: 'Apply Email Template?',
                html: `
                    <div class="text-start">
                        <p class="mb-3">This will apply the <strong>"${templateName}"</strong> template to your email headers and footers.</p>
                        <div class="alert alert-warning mb-3">
                            <i class="fa fa-exclamation-triangle me-2"></i>
                            <strong>Warning:</strong> This will replace your current custom header and footer content.
                        </div>
                        <p class="mb-0">The following will happen:</p>
                        <ul class="list-unstyled mt-2">
                            <li><i class="fa fa-check text-success me-2"></i>Template header will be loaded into custom header</li>
                            <li><i class="fa fa-check text-success me-2"></i>Template footer will be loaded into custom footer</li>
                            <li><i class="fa fa-check text-success me-2"></i>Custom header/footer option will be enabled</li>
                        </ul>
                    </div>
                `,
                icon: 'question',
                showCloseButton: false,
                showCancelButton: true,
                showConfirmButton: true,
                focusCancel: true,
                cancelButtonText: 'Cancel',
                confirmButtonText: 'Apply Template',
                buttonsStyling: false,
                customClass: {
                    confirmButton: 'btn btn-success me-2',
                    cancelButton: 'btn btn-secondary'
                },
                showClass: {
                    popup: 'animate__animated animate__fadeIn'
                },
                hideClass: {
                    popup: 'animate__animated animate__fadeOut'
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    applyEmailTemplate(templateId, templateName);
                }
            });
        }

        function applyEmailTemplate(templateId, templateName) {
            // Show loading state
            Swal.fire({
                title: 'Applying Template...',
                text: 'Please wait while the template is being applied.',
                icon: 'info',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                },
                showClass: {
                    popup: 'animate__animated animate__fadeIn'
                },
                hideClass: {
                    popup: 'animate__animated animate__fadeOut'
                }
            });

            $.ajax({
                url: json_strings.uri.base + 'apply-email-template.php',
                type: 'POST',
                data: {
                    template_id: templateId,
                    csrf_token: document.getElementById('csrf_token').value
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            title: 'Template Applied Successfully!',
                            text: `The "${templateName}" template has been applied to your email headers and footers.`,
                            icon: 'success',
                            showConfirmButton: true,
                            confirmButtonText: 'OK',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-success'
                            },
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        }).then(() => {
                            // Optionally redirect to header/footer page
                            // window.location.href = json_strings.uri.base + 'email-templates.php?section=header_footer';
                        });
                    } else {
                        Swal.fire({
                            title: 'Error Applying Template',
                            text: response.message || 'An error occurred while applying the template.',
                            icon: 'error',
                            showConfirmButton: true,
                            confirmButtonText: 'OK',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-primary'
                            },
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        });
                    }
                },
                error: function(xhr, status, error) {
                    Swal.fire({
                        title: 'Connection Error',
                        text: 'Could not connect to the server. Please try again.',
                        icon: 'error',
                        showConfirmButton: true,
                        confirmButtonText: 'OK',
                        buttonsStyling: false,
                        customClass: {
                            confirmButton: 'btn btn-primary'
                        },
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    });
                }
            });
        }

        // Initialize HTML editors for email template textareas
        var htmlTextareas = document.querySelectorAll('#form_email_template textarea.textarea_high');
        var editors = {};

        htmlTextareas.forEach(function(textarea) {
            if (typeof CodeMirror !== 'undefined') {
                editors[textarea.id] = CodeMirror.fromTextArea(textarea, {
                    mode: 'htmlmixed',
                    lineNumbers: true,
                    lineWrapping: true,
                    autoCloseTags: true,
                    autoCloseBrackets: true,
                    matchBrackets: true,
                    theme: 'default',
                    extraKeys: {
                        "Ctrl-Space": "autocomplete",
                        "F11": function(cm) {
                            cm.setOption("fullScreen", !cm.getOption("fullScreen"));
                        },
                        "Esc": function(cm) {
                            if (cm.getOption("fullScreen")) cm.setOption("fullScreen", false);
                        }
                    }
                });
            }
        });

        // Check if each tag is used or not
        var tags_dt = document.querySelectorAll('#email_available_tags dt button');
        var tags = [];
        Array.prototype.forEach.call(tags_dt, function(tag) {
            tags.push(tag.dataset.tag);
        });

        var textareas = document.querySelectorAll('#form_email_template textarea');

        const check_tags_usage = setInterval(() => {
            tags.forEach(tag => {
                checkTagsUsage(tag);
            });
        }, 1000);

        function checkTagsUsage(tag)
        {
            textareas.forEach(element => {
                const el = document.querySelector('button[data-tag="'+tag+'"]');
                if (!element.value.includes(tag)) {
                    el.classList.add('btn-warning');
                    el.classList.remove('btn-pslight');
                } else {
                    el.classList.add('btn-pslight');
                    el.classList.remove('btn-warning');
                }
            });
        }
    };
})();
(function () {
    'use strict';

    admin.pages.fileEditor = function () {
        var form = document.getElementById('files');
        var formChanged = false;
        var formSubmitting = false;
        var initialFormData = form ? new FormData(form) : null;

        $(document).ready(function(){
            // Datepicker
            if ( $.isFunction($.fn.datepicker) ) {
                $('.date-container .date-field').datepicker({
                    format : 'dd-mm-yyyy',
                    autoclose : true,
                    todayHighlight : true
                });
            }

            // Validation
            var validator = $("#files").validate({
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });

            var file = $('input[name^="file"]');

            file.filter('input[name$="[name]"]').each(function() {
                $(this).rules("add", {
                    required: true,
                    messages: {
                        required: json_strings.validation.no_name
                    }
                });
            });

            // Copy settings to other files
            function copySettingsToCheckboxes(el, to, question)
            {
                if ( confirm( question ) ) {
                    $(to).each(function(i, obj) {
                        var from_element = document.getElementById($(el).data('copy-from'));
                        $(this).prop('checked', from_element.checked);
                    });
                }
            }

            $('.copy-expiration-settings').on('click', function() {
                copySettingsToCheckboxes($(this), '.checkbox_setting_expires', json_strings.translations.upload_form.copy_expiration);
                // Copy date
                var element = $('#'+$(this).data('copy-date-from'));
                var date = element.val();
                $('.date-field').each(function(i, obj) {
                    console.log(date);
                    $('.date-field').datepicker('update', date);
                });
            });

            $('.copy-public-settings').on('click', function() {
                copySettingsToCheckboxes($(this), '.checkbox_setting_public', json_strings.translations.upload_form.copy_public);
            });

            $('.copy-hidden-settings').on('click', function() {
                copySettingsToCheckboxes($(this), '.checkbox_setting_hidden', json_strings.translations.upload_form.copy_hidden);
            });

            // Download limit settings toggle
            $('.checkbox_download_limit_enabled').on('change', function() {
                var settings = $(this).closest('.file_data').find('.download_limit_settings');
                if ($(this).is(':checked')) {
                    settings.slideDown();
                } else {
                    settings.slideUp();
                }
            });

            // Copy download limit settings (legacy - kept for backwards compatibility)
            $('.copy-download-limit-settings').on('click', function() {
                if (confirm(json_strings.translations.upload_form.copy_download_limits || 'Apply these download limit settings to all files?')) {
                    var from_element = document.getElementById($(this).data('copy-from'));
                    var from_wrapper = $(from_element).closest('.file_data');

                    // Copy enabled checkbox state
                    $('.checkbox_download_limit_enabled').each(function(i, obj) {
                        $(this).prop('checked', from_element.checked);
                        // Show/hide settings based on checkbox
                        var settings = $(this).closest('.file_data').find('.download_limit_settings');
                        if (from_element.checked) {
                            settings.show();
                        } else {
                            settings.hide();
                        }
                    });

                    // Copy limit type
                    var from_type = from_wrapper.find('input[type="radio"]:checked').val();
                    $('input[name$="[download_limit_type]"]').each(function() {
                        if ($(this).val() === from_type) {
                            $(this).prop('checked', true);
                        }
                    });

                    // Copy limit count
                    var from_count = from_wrapper.find('input[type="number"]').val();
                    $('input[name$="[download_limit_count]"]').val(from_count);
                }
            });

            // ===== BULK ACTIONS PANEL =====

            // Toggle bulk actions panel
            $('#toggleBulkActions, .bulk-actions-header').on('click', function(e) {
                if ($(e.target).is('select') || $(e.target).is('button:not(#toggleBulkActions)')) {
                    return; // Don't toggle if clicking on controls
                }
                $('#bulkActionsPanel').toggleClass('collapsed');
            });

            // Helper function to get source file index
            function getSourceFileIndex(selectId) {
                var sourceIndex = $('#' + selectId).val();
                if (!sourceIndex) {
                    alert('Please select a source file first.');
                    return null;
                }
                return parseInt(sourceIndex);
            }

            // Helper function to get file editor wrapper by index
            function getFileWrapperByIndex(index) {
                return $('.file_editor_wrapper').eq(index - 1);
            }

            // Copy all settings from selected file
            $('#bulkCopyAllSettings').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkCopySourceFile');
                if (!sourceIndex) return;

                if (!confirm('Copy all settings from the selected file to all other files?')) {
                    return;
                }

                var sourceWrapper = getFileWrapperByIndex(sourceIndex);

                // Copy expiration
                bulkCopyExpiration(sourceWrapper);

                // Copy download limits
                bulkCopyDownloadLimits(sourceWrapper);

                // Copy public settings
                bulkCopyPublic(sourceWrapper);

                // Copy assignments
                bulkCopyClients(sourceWrapper);
                bulkCopyGroups(sourceWrapper);
                bulkCopyHidden(sourceWrapper);

                // Copy organization
                bulkCopyCategories(sourceWrapper);
                bulkCopyFolder(sourceWrapper);

                alert('All settings copied successfully!');
            });

            // Individual bulk copy functions
            function bulkCopyExpiration(sourceWrapper) {
                var sourceCheckbox = sourceWrapper.find('.checkbox_setting_expires');
                var sourceDateField = sourceWrapper.find('.date-field');

                if (sourceCheckbox.length) {
                    var isChecked = sourceCheckbox.is(':checked');
                    var dateValue = sourceDateField.val();

                    $('.checkbox_setting_expires').prop('checked', isChecked);
                    $('.date-field').datepicker('update', dateValue);
                }
            }

            function bulkCopyDownloadLimits(sourceWrapper) {
                var sourceCheckbox = sourceWrapper.find('.checkbox_download_limit_enabled');

                if (sourceCheckbox.length) {
                    var isEnabled = sourceCheckbox.is(':checked');
                    var limitType = sourceWrapper.find('input[name$="[download_limit_type]"]:checked').val();
                    var limitCount = sourceWrapper.find('input[name$="[download_limit_count]"]').val();

                    $('.checkbox_download_limit_enabled').prop('checked', isEnabled).trigger('change');

                    $('input[name$="[download_limit_type]"]').each(function() {
                        if ($(this).val() === limitType) {
                            $(this).prop('checked', true);
                        }
                    });

                    $('input[name$="[download_limit_count]"]').val(limitCount);
                }
            }

            function bulkCopyPublic(sourceWrapper) {
                var sourceCheckbox = sourceWrapper.find('.checkbox_setting_public');

                if (sourceCheckbox.length) {
                    var isChecked = sourceCheckbox.is(':checked');
                    $('.checkbox_setting_public').prop('checked', isChecked);
                }
            }

            function bulkCopyClients(sourceWrapper) {
                var sourceSelect = sourceWrapper.find('.assignments_clients');

                if (sourceSelect.length) {
                    var selectedValues = sourceSelect.val() || [];

                    $('.assignments_clients').each(function() {
                        $(this).val(selectedValues).trigger('change');
                    });
                }
            }

            function bulkCopyGroups(sourceWrapper) {
                var sourceSelect = sourceWrapper.find('.assignments_groups');

                if (sourceSelect.length) {
                    var selectedValues = sourceSelect.val() || [];

                    $('.assignments_groups').each(function() {
                        $(this).val(selectedValues).trigger('change');
                    });
                }
            }

            function bulkCopyHidden(sourceWrapper) {
                var sourceCheckbox = sourceWrapper.find('.checkbox_setting_hidden');

                if (sourceCheckbox.length) {
                    var isChecked = sourceCheckbox.is(':checked');
                    $('.checkbox_setting_hidden').prop('checked', isChecked);
                }
            }

            function bulkCopyCategories(sourceWrapper) {
                var sourceSelect = sourceWrapper.find('select[name$="[categories][]"]');

                if (sourceSelect.length) {
                    var selectedValues = sourceSelect.val() || [];

                    $('select[name$="[categories][]"]').each(function() {
                        $(this).val(selectedValues).trigger('change');
                    });
                }
            }

            function bulkCopyFolder(sourceWrapper) {
                var sourceSelect = sourceWrapper.find('select[name$="[folder_id]"]');

                if (sourceSelect.length) {
                    var selectedValue = sourceSelect.val();

                    $('select[name$="[folder_id]"]').each(function() {
                        $(this).val(selectedValue).trigger('change');
                    });
                }
            }

            // Individual bulk action buttons
            $('.bulk-copy-expiration').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkExpirationSource');
                if (!sourceIndex) return;

                if (confirm('Copy expiration settings from the selected file to all other files?')) {
                    bulkCopyExpiration(getFileWrapperByIndex(sourceIndex));
                    alert('Expiration settings copied!');
                }
            });

            $('.bulk-copy-download-limits').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkDownloadLimitSource');
                if (!sourceIndex) return;

                if (confirm('Copy download limit settings from the selected file to all other files?')) {
                    bulkCopyDownloadLimits(getFileWrapperByIndex(sourceIndex));
                    alert('Download limit settings copied!');
                }
            });

            $('.bulk-copy-public').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkVisibilitySource');
                if (!sourceIndex) return;

                if (confirm('Copy public visibility settings from the selected file to all other files?')) {
                    bulkCopyPublic(getFileWrapperByIndex(sourceIndex));
                    alert('Visibility settings copied!');
                }
            });

            $('.bulk-copy-clients').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkAssignmentSource');
                if (!sourceIndex) return;

                if (confirm('Copy client assignments from the selected file to all other files?')) {
                    bulkCopyClients(getFileWrapperByIndex(sourceIndex));
                    alert('Client assignments copied!');
                }
            });

            $('.bulk-copy-groups').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkAssignmentSource');
                if (!sourceIndex) return;

                if (confirm('Copy group assignments from the selected file to all other files?')) {
                    bulkCopyGroups(getFileWrapperByIndex(sourceIndex));
                    alert('Group assignments copied!');
                }
            });

            $('.bulk-copy-hidden').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkAssignmentSource');
                if (!sourceIndex) return;

                if (confirm('Copy hidden status from the selected file to all other files?')) {
                    bulkCopyHidden(getFileWrapperByIndex(sourceIndex));
                    alert('Hidden status copied!');
                }
            });

            $('.bulk-copy-categories').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkOrganizationSource');
                if (!sourceIndex) return;

                if (confirm('Copy category assignments from the selected file to all other files?')) {
                    bulkCopyCategories(getFileWrapperByIndex(sourceIndex));
                    alert('Category assignments copied!');
                }
            });

            $('.bulk-copy-folder').on('click', function() {
                var sourceIndex = getSourceFileIndex('bulkOrganizationSource');
                if (!sourceIndex) return;

                if (confirm('Copy folder assignment from the selected file to all other files?')) {
                    bulkCopyFolder(getFileWrapperByIndex(sourceIndex));
                    alert('Folder assignment copied!');
                }
            });

            // Collapse - expand single item
            $('.toggle_file_editor').on('click', function(e) {
                let wrapper = $(this).parents('.file_editor_wrapper');
                wrapper.toggleClass('collapsed');
            });

            // Collapse all
            document.getElementById('files_collapse_all').addEventListener('click', function(e) {
                let wrappers = document.querySelectorAll('.file_editor_wrapper');
                wrappers.forEach(wrapper => {
                    wrapper.classList.add('collapsed');
                });
                    
            })

            // Expand all
            document.getElementById('files_expand_all').addEventListener('click', function(e) {
                let wrappers = document.querySelectorAll('.file_editor_wrapper');
                wrappers.forEach(wrapper => {
                    wrapper.classList.remove('collapsed');
                });
                    
            })
        });

        // Track form changes for unsaved changes warning
        if (form) {
            // Track changes to all form inputs
            form.addEventListener('change', function() {
                formChanged = true;
            });

            // Also track text input changes
            var textInputs = form.querySelectorAll('input[type="text"], input[type="email"], input[type="password"], input[type="number"], input[type="checkbox"], textarea, select');
            textInputs.forEach(function(input) {
                input.addEventListener('input', function() {
                    // Check if form has actually changed from initial state
                    var currentFormData = new FormData(form);
                    formChanged = !areFormDataEqual(initialFormData, currentFormData);
                });
            });

            // Set flag when form is being submitted
            form.addEventListener('submit', function() {
                formSubmitting = true;
            });

            // Helper function to compare FormData objects
            function areFormDataEqual(formData1, formData2) {
                if (!formData1 || !formData2) return false;

                var entries1 = Array.from(formData1.entries());
                var entries2 = Array.from(formData2.entries());

                if (entries1.length !== entries2.length) {
                    return false;
                }

                for (var i = 0; i < entries1.length; i++) {
                    if (entries1[i][0] !== entries2[i][0] || entries1[i][1] !== entries2[i][1]) {
                        return false;
                    }
                }

                return true;
            }

            // Warn user before leaving if there are unsaved changes
            window.addEventListener('beforeunload', function(e) {
                if (formChanged && !formSubmitting) {
                    var confirmationMessage = 'You have unsaved changes. Are you sure you want to leave this page?';
                    e.returnValue = confirmationMessage;
                    return confirmationMessage;
                }
            });

            // Handle navigation away from the page (for links within the application)
            document.addEventListener('click', function(e) {
                // Check if it's a link that would navigate away
                var target = e.target.closest('a');
                if (target && target.href && !target.href.startsWith('#') && !target.classList.contains('delete-confirm')) {
                    if (formChanged && !formSubmitting) {
                        e.preventDefault();
                        var targetUrl = target.href;

                        Swal.fire({
                            title: 'Unsaved Changes',
                            text: 'You have unsaved changes. Are you sure you want to leave this page?',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#d33',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Yes, leave page',
                            cancelButtonText: 'Stay on page',
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        }).then(function(result) {
                            if (result.isConfirmed) {
                                formSubmitting = true; // Prevent the beforeunload warning
                                window.location.href = targetUrl;
                            }
                        });
                    }
                }
            });
        }
    };
})();
(function () {
    'use strict';

    admin.pages.groupForm = function () {
        var form = document.getElementById('group_form');
        var formChanged = false;
        var formSubmitting = false;
        var initialFormData = form ? new FormData(form) : null;

        $(document).ready(function(){
            var validator = $("#group_form").validate({
                rules: {
                    name: {
                        required: true
                    },
                },
                messages: {
                    name: {
                        required: json_strings.validation.no_name
                    },
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });
        });

        // Track form changes for unsaved changes warning
        if (form) {
            // Track changes to all form inputs
            form.addEventListener('change', function() {
                formChanged = true;
            });

            // Also track text input changes
            var textInputs = form.querySelectorAll('input[type="text"], input[type="email"], input[type="password"], input[type="number"], input[type="checkbox"], textarea, select');
            textInputs.forEach(function(input) {
                input.addEventListener('input', function() {
                    // Check if form has actually changed from initial state
                    var currentFormData = new FormData(form);
                    formChanged = !areFormDataEqual(initialFormData, currentFormData);
                });
            });

            // Set flag when form is being submitted
            form.addEventListener('submit', function() {
                formSubmitting = true;
            });

            // Helper function to compare FormData objects
            function areFormDataEqual(formData1, formData2) {
                if (!formData1 || !formData2) return false;

                var entries1 = Array.from(formData1.entries());
                var entries2 = Array.from(formData2.entries());

                if (entries1.length !== entries2.length) {
                    return false;
                }

                for (var i = 0; i < entries1.length; i++) {
                    if (entries1[i][0] !== entries2[i][0] || entries1[i][1] !== entries2[i][1]) {
                        return false;
                    }
                }

                return true;
            }

            // Warn user before leaving if there are unsaved changes
            window.addEventListener('beforeunload', function(e) {
                if (formChanged && !formSubmitting) {
                    var confirmationMessage = 'You have unsaved changes. Are you sure you want to leave this page?';
                    e.returnValue = confirmationMessage;
                    return confirmationMessage;
                }
            });

            // Handle navigation away from the page (for links within the application)
            document.addEventListener('click', function(e) {
                // Check if it's a link that would navigate away
                var target = e.target.closest('a');
                if (target && target.href && !target.href.startsWith('#') && !target.classList.contains('delete-confirm')) {
                    if (formChanged && !formSubmitting) {
                        e.preventDefault();
                        var targetUrl = target.href;

                        Swal.fire({
                            title: 'Unsaved Changes',
                            text: 'You have unsaved changes. Are you sure you want to leave this page?',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#d33',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Yes, leave page',
                            cancelButtonText: 'Stay on page',
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        }).then(function(result) {
                            if (result.isConfirmed) {
                                formSubmitting = true; // Prevent the beforeunload warning
                                window.location.href = targetUrl;
                            }
                        });
                    }
                }
            });
        }
    };
})();
(function () {
    'use strict';

    admin.pages.import_external = function () {
        const selectAllCheckbox = document.getElementById('select_all_checkbox');
        const fileCheckboxes = document.querySelectorAll('.file_checkbox');
        const selectAllBtn = document.getElementById('select_all');
        const selectNoneBtn = document.getElementById('select_none');

        // Handle select all checkbox
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                fileCheckboxes.forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
            });
        }

        // Handle select all button
        if (selectAllBtn) {
            selectAllBtn.addEventListener('click', function() {
                fileCheckboxes.forEach(checkbox => {
                    checkbox.checked = true;
                });
                if (selectAllCheckbox) selectAllCheckbox.checked = true;
            });
        }

        // Handle select none button
        if (selectNoneBtn) {
            selectNoneBtn.addEventListener('click', function() {
                fileCheckboxes.forEach(checkbox => {
                    checkbox.checked = false;
                });
                if (selectAllCheckbox) selectAllCheckbox.checked = false;
            });
        }

        // Update select all checkbox when individual checkboxes change
        fileCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                if (selectAllCheckbox) {
                    const checkedCount = document.querySelectorAll('.file_checkbox:checked').length;
                    selectAllCheckbox.checked = (checkedCount === fileCheckboxes.length);
                    selectAllCheckbox.indeterminate = (checkedCount > 0 && checkedCount < fileCheckboxes.length);
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.importOrphans = function () {

        $(document).ready(function(){
            $("#import_orphans").submit(function() {
				var checks = $("td>input:checkbox").serializeArray(); 
				
				if (checks.length == 0) { 
					alert(json_strings.translations.select_one_or_more);
					return false; 
				} 
			});
			
			/**
			 * Only select the current file when clicking an "edit" button
			 */
			$('.btn-edit-file').click(function(e) {
				$('#select_all').prop('checked', false);
				$('td .select_file_checkbox').prop('checked', false);
				$(this).parents('tr').find('td .select_file_checkbox').prop('checked', true);
				$("#import_orphans").submit();
			});
        });
    };
})();
(function () {
    'use strict';

    admin.pages.install = function () {

        $(document).ready(function(){
            var validator = $("#install_form").validate({
                rules: {
                    install_title: {
                        required: true,
                    },
                    base_uri: {
                        required: true
                        // url: true // Does not work on localhost
                    },
                    admin_name: {
                        required: true,
                    },
                    admin_email: {
                        required: true,
                        email: true
                    },
                    admin_username: {
                        required: true,
                        minlength: json_strings.character_limits.user_min,
                        maxlength: json_strings.character_limits.user_max,
                        alphanumericUsername: true
                    },
                    admin_pass: {
                        required: true,
                        minlength: json_strings.character_limits.password_min,
                        maxlength: json_strings.character_limits.password_max,
                        passwordValidCharacters: true
                    },
                },
                messages: {
                    category_name: {
                        required: json_strings.validation.no_name,
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                },
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.integration_form = function () {
        // For integrations-add.php
        const typeSelect = document.getElementById('type');
        if (typeSelect) {
            const credentialsContainer = document.getElementById('credentials_fields');
            const integrationInfos = document.querySelectorAll('.integration-type-info');
            const defaultInfo = document.getElementById('default_info');

            // Available types configuration - will be populated by PHP
            const availableTypesElement = document.getElementById('integration_form');
            const availableTypes = availableTypesElement ? JSON.parse(availableTypesElement.dataset.availableTypes) : {};

            function updateCredentialFields() {
                const selectedType = typeSelect.value;
                credentialsContainer.innerHTML = '';

                // Hide all integration info panels
                integrationInfos.forEach(info => info.style.display = 'none');

                if (selectedType && availableTypes[selectedType]) {
                    const typeConfig = availableTypes[selectedType];

                    // Show relevant integration info
                    const typeInfo = document.querySelector(`[data-type="${selectedType}"]`);
                    if (typeInfo) {
                        typeInfo.style.display = 'block';
                        defaultInfo.style.display = 'none';
                    }

                    // Don't create fields for coming soon types
                    if (typeConfig.coming_soon) {
                        return;
                    }

                    if (typeConfig.fields) {
                        // Create header
                        const header = document.createElement('div');
                        header.className = 'mb-4';
                        header.innerHTML = `
                            <h5>${typeConfig.name} Configuration</h5>
                            <hr>
                        `;
                        credentialsContainer.appendChild(header);

                        // Create fields
                        Object.keys(typeConfig.fields).forEach(fieldName => {
                            const fieldConfig = typeConfig.fields[fieldName];
                            const fieldContainer = document.createElement('div');
                            fieldContainer.className = 'mb-3';
                            let fieldHtml = '';
                            const fieldId = `credentials_${fieldName}`;
                            const fieldNameAttr = `credentials[${fieldName}]`;
                            const isRequired = fieldConfig.required ? 'required' : '';
                            const requiredMark = fieldConfig.required ? ' *' : '';

                            if (fieldConfig.type === 'select' && fieldConfig.options) {
                                fieldHtml = `
                                    <label for="${fieldId}" class="form-label">${fieldConfig.label}${requiredMark}</label>
                                    <select name="${fieldNameAttr}" id="${fieldId}" class="form-select" ${isRequired}>
                                        <option value="">Choose...</option>
                                `;
                                Object.keys(fieldConfig.options).forEach(optionValue => {
                                    fieldHtml += `<option value="${optionValue}">${fieldConfig.options[optionValue]}</option>`;
                                });
                                fieldHtml += '</select>';
                            } else {
                                const inputType = fieldConfig.type === 'password' ? 'password' : 'text';
                                fieldHtml = `
                                    <label for="${fieldId}" class="form-label">${fieldConfig.label}${requiredMark}</label>
                                    <input type="${inputType}" name="${fieldNameAttr}" id="${fieldId}"
                                           class="form-control" ${isRequired}>
                                `;
                            }

                            fieldContainer.innerHTML = fieldHtml;
                            credentialsContainer.appendChild(fieldContainer);
                        });
                    }
                } else {
                    // Show default info when no type selected
                    if (defaultInfo) {
                        defaultInfo.style.display = 'block';
                    }
                }
            }

            // Update fields when type changes
            typeSelect.addEventListener('change', updateCredentialFields);

            // Initialize on page load
            updateCredentialFields();
        }

        // For integrations-edit.php
        const updateCredentialsCheckbox = document.getElementById('update_credentials');
        const credentialsSection = document.getElementById('credentials_section');

        if (updateCredentialsCheckbox && credentialsSection) {
            updateCredentialsCheckbox.addEventListener('change', function() {
                if (this.checked) {
                    credentialsSection.style.display = 'block';
                } else {
                    credentialsSection.style.display = 'none';
                }
            });
        }
    };
})();
(function () {
    'use strict';

    admin.pages.loginForm = function () {

        $(document).ready(function(){
            // let data_changed = false;
            // $('body').on('input', '#username, #password', function(e) {
            //     data_changed = true;
            // });

            // $('body').on('change', '#language', function (e) {
            //     if (data_changed == false) {
            //         let change_locale_uri = json_strings.uri.base + 'process.php?do=change_language&language=' + $(this).val();
            //         window.location.replace(change_locale_uri);;
            //     }
            // });

            var initial = $('.seconds_countdown').text();
            if (initial) {
                $('#btn_submit').attr('disabled', 'disabled');
            }
            var downloadTimer = setInterval(function(){
                if (initial <= 0) {
                    clearInterval(downloadTimer);
                    $('#btn_submit').removeAttr('disabled');
                    $('#message_countdown').slideUp();
                }
                $('.seconds_countdown').text(initial);
                initial -= 1;
            }, 1000);

            var validator = $("#login_form").validate({
                rules: {
                    username: {
                        required: true,
                    },
                    password: {
                        required: true,
                    },
                },
                messages: {
                    username: {
                        required: json_strings.validation.no_user,
                    },
                    password: {
                        required: json_strings.validation.no_pass,
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                },
                submitHandler: function(form) {
                    form.submit();

                    var button_loading_text = json_strings.login.logging_in;
                    $('#btn_submit').html('<i class="fa fa-cog fa-spin fa-fw"></i><span class="sr-only"></span> '+button_loading_text+'...').attr('disabled', 'disabled');
                    /*
                    
                    var button_text = json_strings.login.button_text;
                    var button_redirecting_text = json_strings.login.redirecting;
                    var url = $(form).attr('action');
                    $('.ajax_response').html('').removeClass('alert alert-danger alert-success').slideUp();
                    

                    $.ajax({
                        cache: false,
                        method: 'POST',
                        url: url,
                        data: $(form).serialize(),
                    }).done( function(data) {
                        var obj = JSON.parse(data);
                        if ( obj.status == 'success' ) {
                            $('#submit').html('<i class="fa fa-check"></i><span class="sr-only"></span> '+button_redirecting_text+'...');
                            $('#submit').removeClass('btn-primary').addClass('btn-success');
                            setTimeout('window.location.href = "'+obj.location+'"', 1000);
                        } else {
                            $('.ajax_response').addClass('alert alert-danger').slideDown().html(obj.message);
                            $('#submit').html(button_text).removeAttr('disabled');
                        }
                    }).fail( function(data) {
                        $('.ajax_response').addClass('alert alert-danger').slideDown().html('Uncaught error');
                        $('#submit').html(button_text).removeAttr('disabled');
                    }).always( function() {
                    });

                    return false;
                    */
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.loginLdapForm = function () {

        $(document).ready(function(){
            var validator = $("#login_ldap_form").validate({
                rules: {
                    ldap_email: {
                        required: true,
                    },
                    ldap_password: {
                        required: true,
                    },
                },
                messages: {
                    ldap_email: {
                        required: json_strings.validation.no_email,
                    },
                    ldap_password: {
                        required: json_strings.validation.no_pass,
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                },
                submitHandler: function(form) {
                    form.submit();

                    var button_loading_text = json_strings.login.logging_in;
                    $('#ldap_submit').html('<i class="fa fa-cog fa-spin fa-fw"></i><span class="sr-only"></span> '+button_loading_text+'...').attr('disabled', 'disabled');
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.options = function () {
        var tagifyContainer = document.getElementById('allowed_file_types');
        var tagify = new Tagify (tagifyContainer);
        //tagifyContainer.addEventListener('change', tagifyOnChange)

        function tagifyOnChange(e){
            console.log(e.target.value)
        }

        $(document).ready(function(){
            var validator = $("#options").validate({
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                },
            });

            $('#download_method').on('change', function(e) {
                var method = $(this).find('option:selected').val();
                $('.method_note').hide();
                $('.method_note[data-method="'+method+'"]').show();
            });

            $('#download_method').trigger('change');
        });

        const captchaMethodSelect = document.getElementById('captcha_method');
        const captchaOptionsBlocks = document.querySelectorAll('.captcha_options_block');
        if (elementExists(captchaMethodSelect)) {
            captchaMethodSelect.addEventListener('change', function(e) {
                const showOptionsBlock = document.getElementById('captcha_' + e.target.value)
                console.log(showOptionsBlock);
                for (let i = 0; i < captchaOptionsBlocks.length; i++) {
                    const captchaOptionsBlock = captchaOptionsBlocks[i];
                    captchaOptionsBlock.classList.add('d-none');
                }

                if (elementExists(showOptionsBlock)) {
                    showOptionsBlock.classList.remove('d-none');
                }
            });
        }

        // Mail system options visibility control
        const mailSystemSelect = document.getElementById('mail_system_use');
        if (elementExists(mailSystemSelect)) {
            function toggleMailFields() {
                const selectedValue = mailSystemSelect.value;
                const authFields = document.querySelectorAll('.mail-auth-field');
                const smtpFields = document.querySelectorAll('.mail-smtp-field');

                // Find SMTP section header
                const h3Elements = document.querySelectorAll('h3');
                let smtpSection = null;
                h3Elements.forEach(h3 => {
                    if (h3.textContent.includes('SMTP options')) {
                        smtpSection = h3;
                    }
                });

                // Hide all conditional fields initially
                authFields.forEach(field => field.style.display = 'none');
                smtpFields.forEach(field => field.style.display = 'none');

                // Hide SMTP section header
                if (smtpSection) {
                    smtpSection.style.display = 'none';
                }

                // Show username/password fields for SMTP and Gmail
                if (selectedValue === 'smtp' || selectedValue === 'gmail') {
                    authFields.forEach(field => field.style.display = '');
                }

                // Show SMTP-specific fields and section only for SMTP
                if (selectedValue === 'smtp') {
                    smtpFields.forEach(field => field.style.display = '');
                    if (smtpSection) {
                        smtpSection.style.display = '';
                    }
                }
            }

            // Set initial state
            toggleMailFields();

            // Listen for changes
            mailSystemSelect.addEventListener('change', toggleMailFields);
        }

        // Branding section - Logo preview and upload functionality
        const logoPreview = document.getElementById('logo-preview-img');
        if (elementExists(logoPreview)) {
            // Debug SVG visibility
            console.log('Logo preview element found:', logoPreview);
            console.log('Logo src:', logoPreview.src);

            // Add error handling for logo loading
            logoPreview.addEventListener('load', function() {
                console.log('Logo loaded successfully');
            });

            logoPreview.addEventListener('error', function() {
                console.error('Logo failed to load');
                console.error('Failed URL:', logoPreview.src);
            });
        }

        const logoInput = document.getElementById('select_logo');
        const logoWarning = document.getElementById('logo-upload-warning');
        if (elementExists(logoInput) && elementExists(logoPreview)) {
            logoInput.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    // Validate file type
                    const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/svg+xml'];
                    if (!validTypes.includes(file.type)) {
                        alert('Please select a valid image file (JPG, PNG, GIF, SVG)');
                        logoInput.value = '';
                        return;
                    }

                    // Validate file size (10MB)
                    if (file.size > 10 * 1024 * 1024) {
                        alert('File size must be less than 10MB');
                        logoInput.value = '';
                        return;
                    }

                    // Show preview
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        logoPreview.src = e.target.result;
                        logoPreview.classList.add('preview-selected');
                        if (elementExists(logoWarning)) {
                            logoWarning.style.display = 'block';
                        }
                    };
                    reader.readAsDataURL(file);
                } else {
                    // Reset preview if no file selected
                    logoPreview.classList.remove('preview-selected');
                    if (elementExists(logoWarning)) {
                        logoWarning.style.display = 'none';
                    }
                }
            });
        }

        // Branding section - Favicon preview and upload functionality
        const faviconInput = document.getElementById('select_favicon');
        const faviconPreview = document.getElementById('favicon-preview-img');
        const faviconWarning = document.getElementById('favicon-upload-warning');
        if (elementExists(faviconInput) && elementExists(faviconPreview)) {
            faviconInput.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    // Validate file type
                    const validTypes = ['image/x-icon', 'image/vnd.microsoft.icon', 'image/png', 'image/gif', 'image/jpeg', 'image/jpg', 'image/svg+xml'];
                    if (!validTypes.includes(file.type)) {
                        alert('Please select a valid favicon file (ICO, PNG, GIF, JPG, SVG)');
                        faviconInput.value = '';
                        return;
                    }

                    // Validate file size (1MB)
                    if (file.size > 1024 * 1024) {
                        alert('Favicon file size must be less than 1MB');
                        faviconInput.value = '';
                        return;
                    }

                    // Show preview
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        faviconPreview.src = e.target.result;
                        faviconPreview.classList.add('preview-selected');
                        if (elementExists(faviconWarning)) {
                            faviconWarning.style.display = 'block';
                        }
                    };
                    reader.readAsDataURL(file);
                } else {
                    // Reset preview if no file selected
                    faviconPreview.classList.remove('preview-selected');
                    if (elementExists(faviconWarning)) {
                        faviconWarning.style.display = 'none';
                    }
                }
            });
        }

        // Hide upload warnings when form is submitted
        const form = document.getElementById('options');
        if (elementExists(form)) {
            form.addEventListener('submit', function() {
                if (elementExists(logoWarning)) {
                    logoWarning.style.display = 'none';
                }
                if (elementExists(faviconWarning)) {
                    faviconWarning.style.display = 'none';
                }
            });
        }

        // Options section navigation - scroll tracking and active state
        const optionsNav = document.querySelector('.options-section-nav');
        if (elementExists(optionsNav)) {
            const navLinks = optionsNav.querySelectorAll('.nav-link');
            const sections = [];

            // Build sections array from nav links
            navLinks.forEach(link => {
                const href = link.getAttribute('href');
                if (href && href.startsWith('#')) {
                    const sectionId = href.substring(1);
                    const sectionElement = document.getElementById(sectionId);
                    if (sectionElement) {
                        sections.push({
                            id: sectionId,
                            element: sectionElement,
                            navLink: link
                        });
                    }
                }
            });

            // Function to update active nav item based on scroll position
            function updateActiveNav() {
                // Get current scroll position with offset for sticky nav (90px from top + nav height)
                const scrollPosition = window.scrollY + 120;

                let activeSection = null;

                // Find which section is currently in view
                sections.forEach(section => {
                    const sectionTop = section.element.offsetTop;
                    const sectionBottom = sectionTop + section.element.offsetHeight;

                    if (scrollPosition >= sectionTop && scrollPosition < sectionBottom) {
                        activeSection = section;
                    }
                });

                // If we're at the top of the page, activate first section
                if (!activeSection && window.scrollY < 200) {
                    activeSection = sections[0];
                }

                // Update active classes
                if (activeSection) {
                    sections.forEach(section => {
                        if (section.id === activeSection.id) {
                            section.navLink.classList.add('active');
                        } else {
                            section.navLink.classList.remove('active');
                        }
                    });
                }
            }

            // Throttle scroll event for performance
            let scrollTimeout;
            window.addEventListener('scroll', function() {
                if (scrollTimeout) {
                    window.cancelAnimationFrame(scrollTimeout);
                }
                scrollTimeout = window.requestAnimationFrame(function() {
                    updateActiveNav();
                });
            });

            // Handle nav link clicks
            navLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    const href = this.getAttribute('href');
                    if (href && href.startsWith('#')) {
                        const targetId = href.substring(1);
                        const targetElement = document.getElementById(targetId);
                        if (targetElement) {
                            // Scroll to section with offset for sticky nav
                            const offset = 100;
                            const elementPosition = targetElement.offsetTop - offset;
                            window.scrollTo({
                                top: elementPosition,
                                behavior: 'smooth'
                            });

                            // Update active state immediately
                            navLinks.forEach(l => l.classList.remove('active'));
                            this.classList.add('active');
                        }
                    }
                });
            });

            // Initialize active state on page load
            updateActiveNav();
        }

        // Encryption settings - handle dependency between enabled and required
        const encryptionEnabledCheckbox = document.getElementById('files_encryption_enabled');
        const encryptionRequiredCheckbox = document.getElementById('files_encryption_required');

        if (elementExists(encryptionEnabledCheckbox) && elementExists(encryptionRequiredCheckbox)) {
            function toggleEncryptionRequired() {
                if (!encryptionEnabledCheckbox.checked) {
                    encryptionRequiredCheckbox.checked = false;
                    encryptionRequiredCheckbox.disabled = true;
                    // Add visual feedback
                    const label = encryptionRequiredCheckbox.closest('.col-sm-8');
                    if (label) {
                        label.style.opacity = '0.5';
                    }
                } else {
                    encryptionRequiredCheckbox.disabled = false;
                    const label = encryptionRequiredCheckbox.closest('.col-sm-8');
                    if (label) {
                        label.style.opacity = '1';
                    }
                }
            }

            // Set initial state
            toggleEncryptionRequired();

            // Listen for changes
            encryptionEnabledCheckbox.addEventListener('change', toggleEncryptionRequired);
        }
    };
})();
(function () {
    'use strict';

    admin.pages.publicFilesList = function () {

        $(document).ready(function(){
            $('select[name="group"]').on('change', function(e) {
                var token = $(this).find(':selected').data('token');
                $('input[name="token"]').val(token);
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.resetPasswordEnterEmail = function () {

        $(document).ready(function(){
            var validator = $("#reset_password_enter_email").validate({
                rules: {
                    email: {
                        required: true,
                        email: true,
                    },
                },
                messages: {
                    email: {
                        required: json_strings.validation.no_email,
                        email: json_strings.validation.invalid_email
                    },
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.resetPasswordEnterNew = function () {

        $(document).ready(function(){
            var validator = $("#reset_password_enter_new").validate({
                rules: {
                    password: {
                        required: true,
                        minlength: json_strings.character_limits.password_min,
                        maxlength: json_strings.character_limits.password_max,
                        passwordValidCharacters: true
                    }
                },
                messages: {
                    password: {
                        required: json_strings.validation.no_pass,
                        minlength: json_strings.validation.length_pass,
                        maxlength: json_strings.validation.length_pass,
                        passwordValidCharacters: json_strings.validation.alpha_pass
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.rolePermissions = function () {
        const permissionCheckboxes = document.querySelectorAll('.permission-checkbox');
        const selectAllBtn = document.getElementById('select-all');
        const selectNoneBtn = document.getElementById('select-none');
        const categoryToggles = document.querySelectorAll('.category-toggle');

        // Store initial state
        const initialState = Array.from(permissionCheckboxes).map(cb => cb.checked);
        let changeCount = 0;

        function updateCounts() {
            // Calculate changes for unsaved warning functionality
            changeCount = 0;
            permissionCheckboxes.forEach((cb, index) => {
                if (cb.checked !== initialState[index]) {
                    changeCount++;
                }
            });
            // Note: Visual count display removed with Permission Summary box
        }

        // Select all permissions
        if (selectAllBtn) {
            selectAllBtn.addEventListener('click', function(e) {
                e.preventDefault();
                permissionCheckboxes.forEach(cb => {
                    cb.checked = true;
                });
                updateCounts();
            });
        }

        // Select no permissions
        if (selectNoneBtn) {
            selectNoneBtn.addEventListener('click', function(e) {
                e.preventDefault();
                permissionCheckboxes.forEach(cb => {
                    cb.checked = false;
                });
                updateCounts();
            });
        }

        // Category toggles
        categoryToggles.forEach(toggle => {
            toggle.addEventListener('click', function(e) {
                e.preventDefault();
                const category = this.dataset.category;
                const categoryCheckboxes = document.querySelectorAll(`.permission-checkbox[data-category="${category}"]`);
                const allChecked = Array.from(categoryCheckboxes).every(cb => cb.checked);

                categoryCheckboxes.forEach(cb => {
                    cb.checked = !allChecked;
                });
                updateCounts();
            });
        });

        // Update counts when checkboxes change
        permissionCheckboxes.forEach(cb => {
            cb.addEventListener('change', updateCounts);
        });

        // Warn about unsaved changes
        window.addEventListener('beforeunload', function(e) {
            if (changeCount > 0) {
                e.preventDefault();
                e.returnValue = '';
            }
        });

        // Don't warn when submitting form
        const permissionsForm = document.getElementById('permissions_form');
        if (permissionsForm) {
            permissionsForm.addEventListener('submit', function() {
                changeCount = 0; // Reset change count to prevent warning
            });
        }

        // Initial count update
        updateCounts();
    };
})();
(function () {
    'use strict';

    admin.pages.roles = function () {
        let currentRoleId = null;
        let currentRoleName = null;

        function initRoles() {
            console.log('Initializing roles page');
            console.log('Modal element exists:', !!document.getElementById('reassignUsersModal'));
            console.log('Bootstrap available:', typeof bootstrap !== 'undefined');

            // Handle role deletion
            console.log('Setting up role deletion handlers');
            const deleteButtons = document.querySelectorAll('.delete-role');
            console.log('Found delete buttons:', deleteButtons.length);

            deleteButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    console.log('Delete button clicked');
                    const roleId = this.dataset.role;
                    const roleName = this.dataset.name;
                    const userCount = parseInt(this.dataset.userCount);

                    console.log('Role data:', { roleId, roleName, userCount });

                    currentRoleId = roleId;
                    currentRoleName = roleName;

                    if (userCount === 0) {
                        // Direct deletion for roles with no users
                        console.log('Direct deletion for role with 0 users');
                        if (confirm('Are you sure you want to delete the role "' + roleName + '"?\n\nThis action cannot be undone.')) {
                            deleteRole(roleId);
                        }
                    } else {
                        // Show reassignment modal for roles with users
                        console.log('Showing modal for role with users');
                        showReassignmentModal(roleId, roleName);
                    }
                });
            });

            // Handle role selection change in modal
            const roleSelect = document.getElementById('new-role-select');
            const confirmButton = document.getElementById('confirm-role-delete');

            if (roleSelect && confirmButton) {
                roleSelect.addEventListener('change', function() {
                    confirmButton.disabled = !this.value;
                });

                // Handle confirm deletion with reassignment
                confirmButton.addEventListener('click', function() {
                    const newRoleId = roleSelect.value;
                    if (newRoleId && currentRoleId) {
                        deleteRoleWithReassignment(currentRoleId, newRoleId);
                    }
                });
            }
        }

        function showReassignmentModal(roleId, roleName) {
            console.log('Showing reassignment modal for role:', roleId, roleName);

            // Check if modal element exists
            const modalElement = document.getElementById('reassignUsersModal');
            if (!modalElement) {
                console.error('Modal element not found!');
                return;
            }

            // Update modal title
            const titleElement = document.getElementById('reassignUsersModalLabel');
            if (titleElement) {
                titleElement.textContent = 'Delete Role "' + roleName + '" - Reassign Users';
            }

            // Load available roles and users
            Promise.all([
                loadAvailableRoles(roleId),
                loadRoleUsers(roleId)
            ]).then(() => {
                console.log('Data loaded, showing modal');
                // Show modal
                try {
                    const modal = new bootstrap.Modal(modalElement);
                    modal.show();
                } catch (error) {
                    console.error('Error showing modal:', error);
                    // Fallback to jQuery if Bootstrap 5 doesn't work
                    if (typeof $ !== 'undefined') {
                        $(modalElement).modal('show');
                    }
                }
            }).catch(error => {
                console.error('Error loading modal data:', error);
            });
        }

        function loadAvailableRoles(excludeRoleId) {
            return fetch(json_strings.uri.base + 'process.php?do=get_roles_for_reassignment&exclude_role=' + excludeRoleId)
                .then(response => response.json())
                .then(data => {
                    const select = document.getElementById('new-role-select');
                    select.innerHTML = '<option value="">Select a role...</option>';

                    if (data.roles) {
                        data.roles.forEach(role => {
                            const option = document.createElement('option');
                            option.value = role.id;
                            option.textContent = role.name;
                            select.appendChild(option);
                        });
                    }
                })
                .catch(error => {
                    console.error('Error loading roles:', error);
                });
        }

        function loadRoleUsers(roleId) {
            return fetch(json_strings.uri.base + 'process.php?do=get_role_users&role_id=' + roleId)
                .then(response => response.json())
                .then(data => {
                    const usersList = document.getElementById('users-list');

                    if (data.users && data.users.length > 0) {
                        const userItems = data.users.map(user =>
                            `<div class="d-flex align-items-center mb-2">
                                <i class="fa fa-user me-2"></i>
                                <strong>${user.name}</strong>
                                <span class="text-muted ms-2">(${user.user})</span>
                            </div>`
                        ).join('');
                        usersList.innerHTML = userItems;
                    } else {
                        usersList.innerHTML = '<em class="text-muted">No users found</em>';
                    }
                })
                .catch(error => {
                    console.error('Error loading users:', error);
                    document.getElementById('users-list').innerHTML = '<em class="text-danger">Error loading users</em>';
                });
        }

        function deleteRole(roleId) {
            submitRoleDeletion(roleId);
        }

        function deleteRoleWithReassignment(roleId, newRoleId) {
            submitRoleDeletion(roleId, newRoleId);
        }

        function submitRoleDeletion(roleId, newRoleId = null) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = json_strings.uri.base + 'process.php?do=delete_role';

            const existingCsrfInput = document.querySelector('input[name="csrf_token"]');
            const csrfValue = existingCsrfInput ? existingCsrfInput.value : '';

            let formHTML = `
                <input type="hidden" name="role_id" value="${roleId}">
                <input type="hidden" name="csrf_token" value="${csrfValue}">
            `;

            if (newRoleId) {
                formHTML += `<input type="hidden" name="reassign_to_role" value="${newRoleId}">`;
            }

            form.innerHTML = formHTML;
            document.body.appendChild(form);
            form.submit();
        }

        // Check if DOM is already loaded or wait for it
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initRoles);
        } else {
            initRoles();
        }
    };
})();
(function () {
    'use strict';

    admin.pages.rolesAdd = function () {

        document.addEventListener('DOMContentLoaded', function() {
            const roleLevel = document.getElementById('role_level');
            const permissionsPreview = document.getElementById('permissions-preview');
            const previewLevel = document.getElementById('preview-level');
            const permissionsList = document.getElementById('permissions-list');

            // Default permissions for each level (simplified)
            const defaultPermissions = {
                9: ['Full system access', 'Manage users', 'Manage settings', 'Manage roles'],
                8: ['Manage clients', 'Manage files', 'View statistics'],
                7: ['Upload files', 'Manage own files'],
                6: ['Upload files', 'View own files'],
                5: ['Upload files', 'View own files'],
                4: ['Upload files', 'View own files'],
                3: ['Upload files', 'View own files'],
                2: ['Upload files', 'View own files'],
                1: ['Upload files', 'View own files']
            };

            if (roleLevel) {
                roleLevel.addEventListener('change', function() {
                    const level = parseInt(this.value);

                    if (level && defaultPermissions[level]) {
                        previewLevel.textContent = level;

                        const perms = defaultPermissions[level] || ['Basic file access'];
                        permissionsList.innerHTML = perms.map(perm =>
                            '<span class="badge bg-secondary me-1 mb-1">' + perm + '</span>'
                        ).join('');

                        permissionsPreview.style.display = 'block';
                    } else {
                        permissionsPreview.style.display = 'none';
                    }
                });
            }

            // Permission management for role creation
            const permissionCheckboxes = document.querySelectorAll('.permission-checkbox-create');
            const selectAllBtn = document.getElementById('select-all-permissions');
            const selectNoneBtn = document.getElementById('select-none-permissions');
            const categoryToggles = document.querySelectorAll('.category-toggle-create');

            // Select all permissions
            if (selectAllBtn) {
                selectAllBtn.addEventListener('click', function() {
                    permissionCheckboxes.forEach(cb => cb.checked = true);
                });
            }

            // Select no permissions
            if (selectNoneBtn) {
                selectNoneBtn.addEventListener('click', function() {
                    permissionCheckboxes.forEach(cb => cb.checked = false);
                });
            }

            // Category toggles
            categoryToggles.forEach(toggle => {
                toggle.addEventListener('click', function() {
                    const category = this.dataset.category;
                    const categoryCheckboxes = document.querySelectorAll(`.permission-checkbox-create[data-category="${category}"]`);
                    const allChecked = Array.from(categoryCheckboxes).every(cb => cb.checked);

                    categoryCheckboxes.forEach(cb => cb.checked = !allChecked);
                });
            });
        });
    };
})();
(function () {
    'use strict';

    admin.pages.rolesEdit = function () {

        document.addEventListener('DOMContentLoaded', function() {
            const deleteButton = document.getElementById('delete-role');

            if (deleteButton) {
                deleteButton.addEventListener('click', function() {
                    const roleId = this.dataset.roleId || this.getAttribute('data-role-id');

                    if (confirm('Are you sure you want to delete this role?\n\nThis action cannot be undone.')) {
                        // Create form and submit
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = 'process.php';

                        // Get CSRF token from any existing form on the page
                        const existingCsrfInput = document.querySelector('input[name="csrf_token"]');
                        const csrfValue = existingCsrfInput ? existingCsrfInput.value : '';

                        form.innerHTML = `
                            <input type="hidden" name="do" value="delete_role">
                            <input type="hidden" name="role_id" value="${roleId}">
                            <input type="hidden" name="csrf_token" value="${csrfValue}">
                        `;

                        document.body.appendChild(form);
                        form.submit();
                    }
                });
            }
        });
    };
})();
(function () {
    'use strict';

    admin.pages.thumbnailsRegenerate = function () {
        var isProcessing = false;
        var currentBatch = 0;
        var totalFiles = 0;
        var processedFiles = 0;
        var failedFiles = 0;

        $(document).ready(function(){
            // Button click handler for regeneration
            $('#regenerate-btn').on('click', function(e) {
                e.preventDefault();

                var selectedFormats = $('input[name="formats[]"]:checked').length;
                if (selectedFormats === 0) {
                    Swal.fire({
                        title: json_strings.thumbnails_regenerate.no_formats_selected,
                        text: json_strings.thumbnails_regenerate.select_one_format,
                        icon: 'warning',
                        showConfirmButton: true,
                        confirmButtonText: json_strings.translations.ok,
                        buttonsStyling: false,
                        customClass: {
                            confirmButton: 'btn btn-primary'
                        },
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    });
                    return false;
                }

                var width = parseInt($('#thumbnail_width').val());
                var height = parseInt($('#thumbnail_height').val());

                if (width < 50 || width > 1000 || height < 50 || height > 1000) {
                    Swal.fire({
                        title: json_strings.thumbnails_regenerate.invalid_dimensions_title,
                        text: json_strings.thumbnails_regenerate.invalid_dimensions,
                        icon: 'error',
                        showConfirmButton: true,
                        confirmButtonText: json_strings.translations.ok,
                        buttonsStyling: false,
                        customClass: {
                            confirmButton: 'btn btn-primary'
                        },
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    });
                    return false;
                }

                // Show SweetAlert2 confirmation and start processing
                Swal.fire({
                    title: json_strings.thumbnails_regenerate.start_regeneration_title,
                    html: '<div class="text-start">' +
                          '<p class="mb-3">' + json_strings.thumbnails_regenerate.process_will_message + '</p>' +
                          '<ul class="">' +
                          '<li>' + json_strings.thumbnails_regenerate.replace_thumbnails + '</li>' +
                          '<li>' + json_strings.thumbnails_regenerate.take_minutes + '</li>' +
                          '<li>' + json_strings.thumbnails_regenerate.run_background + '</li>' +
                          '<li>' + json_strings.thumbnails_regenerate.keep_page_open + '</li>' +
                          '</ul>' +
                          '</div>',
                    icon: 'warning',
                    showCloseButton: false,
                    showCancelButton: true,
                    showConfirmButton: true,
                    focusCancel: true,
                    cancelButtonText: json_strings.translations.cancel,
                    confirmButtonText: json_strings.thumbnails_regenerate.start_regeneration,
                    buttonsStyling: false,
                    customClass: {
                        confirmButton: 'btn btn-primary me-2',
                        cancelButton: 'btn btn-secondary'
                    },
                    showClass: {
                        popup: 'animate__animated animate__fadeIn'
                    },
                    hideClass: {
                        popup: 'animate__animated animate__fadeOut'
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        startThumbnailRegeneration();
                    }
                });

                return false;
            });
            
            // Date validation for filter form
            $('#filter_start_date, #filter_end_date').on('change', function() {
                var startDate = $('#filter_start_date').val();
                var endDate = $('#filter_end_date').val();
                
                if (startDate && endDate && startDate > endDate) {
                    Swal.fire({
                        title: json_strings.thumbnails_regenerate.invalid_date_range_title,
                        text: json_strings.thumbnails_regenerate.invalid_date_range,
                        icon: 'error',
                        showConfirmButton: true,
                        confirmButtonText: json_strings.translations.ok,
                        buttonsStyling: false,
                        customClass: {
                            confirmButton: 'btn btn-primary'
                        },
                        showClass: {
                            popup: 'animate__animated animate__fadeIn'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOut'
                        }
                    });
                    $(this).val('');
                }
            });
            
            // Update counts when format selection changes (placeholder for future enhancement)
            $('input[name="formats[]"]').on('change', function() {
                // TODO: Update filtered count via AJAX
                console.log('Format selection changed - will implement dynamic count update later');
            });
        });

        function startThumbnailRegeneration() {
            if (isProcessing) {
                return;
            }

            isProcessing = true;
            currentBatch = 0;
            totalFiles = 0;
            processedFiles = 0;
            failedFiles = 0;

            var regenerateButton = $('#regenerate-btn');
            var form = $('#thumbnails-form');

            // Disable form and show progress
            regenerateButton.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> ' + json_strings.thumbnails_regenerate.processing);
            form.find('input, button').prop('disabled', true);

            // Create progress container
            createProgressContainer();

            // Get selected formats
            var selectedFormats = [];
            $('input[name="formats[]"]:checked').each(function() {
                selectedFormats.push($(this).val());
            });

            // Get form data
            var formData = {
                formats: selectedFormats,
                width: parseInt($('#thumbnail_width').val()),
                height: parseInt($('#thumbnail_height').val()),
                start_date: $('input[name="filtered_start_date"]').val() || null,
                end_date: $('input[name="filtered_end_date"]').val() || null
            };

            // Start processing
            processNextBatch(formData);
        }

        function createProgressContainer() {
            var progressHtml = '<div id="thumbnail-progress" class="mt-4">' +
                '<div class="ps-card">' +
                '<div class="ps-card-body">' +
                '<h6>' + json_strings.thumbnails_regenerate.processing_title + '</h6>' +
                '<div class="progress mb-3" style="height: 25px;">' +
                '<div class="progress-bar progress-bar-striped progress-bar-animated" ' +
                'role="progressbar" style="width: 0%;" id="progress-bar">' +
                '<span id="progress-text">0%</span>' +
                '</div>' +
                '</div>' +
                '<div class="row">' +
                '<div class="col-md-4">' +
                '<div class="text-center">' +
                '<div class="h4 text-primary" id="processed-count">0</div>' +
                '<small class="text-muted">' + json_strings.thumbnails_regenerate.processed + '</small>' +
                '</div>' +
                '</div>' +
                '<div class="col-md-4">' +
                '<div class="text-center">' +
                '<div class="h4 text-danger" id="failed-count">0</div>' +
                '<small class="text-muted">' + json_strings.thumbnails_regenerate.failed + '</small>' +
                '</div>' +
                '</div>' +
                '<div class="col-md-4">' +
                '<div class="text-center">' +
                '<div class="h4 text-info" id="total-count">0</div>' +
                '<small class="text-muted">' + json_strings.thumbnails_regenerate.total + '</small>' +
                '</div>' +
                '</div>' +
                '</div>' +
                '<div id="current-file" class="mt-3 text-muted"></div>' +
                '<div id="process-log" class="mt-3" style="max-height: 200px; overflow-y: auto; font-family: monospace; font-size: 12px; background: #f8f9fa; padding: 10px; border-radius: 4px; display: none;"></div>' +
                '<button type="button" class="btn btn-sm btn-secondary mt-2" id="toggle-log">' + json_strings.thumbnails_regenerate.show_log + '</button>' +
                '</div>' +
                '</div>' +
                '</div>';
            
            $('#thumbnails-form').after(progressHtml);

            // Add toggle log functionality
            $('#toggle-log').on('click', function() {
                $('#process-log').toggle();
                $(this).text($('#process-log').is(':visible') ? 
                    json_strings.thumbnails_regenerate.hide_log : 
                    json_strings.thumbnails_regenerate.show_log);
            });
        }

        function processNextBatch(formData) {
            var batchSize = 5; // Process 5 files at a time
            var offset = currentBatch * batchSize;

            // Get files for this batch
            $.ajax({
                url: json_strings.uri.base + 'includes/ajax.process.php?do=thumbnails_regenerate_get_files',
                type: 'POST',
                data: {
                    formats: formData.formats,
                    start_date: formData.start_date,
                    end_date: formData.end_date,
                    batch_size: batchSize,
                    offset: offset,
                    csrf_token: document.getElementById('csrf_token').value
                },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var files = response.data.files;
                        
                        // Update total count on first batch
                        if (currentBatch === 0) {
                            totalFiles = response.data.total;
                            $('#total-count').text(totalFiles);
                            
                            if (totalFiles === 0) {
                                logMessage('No files found to process', 'info');
                                finishProcessing();
                                return;
                            }
                        }

                        if (files.length === 0) {
                            // No more files to process
                            finishProcessing();
                            return;
                        }

                        // Process files in this batch
                        processBatchFiles(files, formData, function() {
                            // Continue with next batch if there are more files
                            if (response.data.has_more) {
                                currentBatch++;
                                processNextBatch(formData);
                            } else {
                                finishProcessing();
                            }
                        });
                    } else {
                        logMessage('Error getting files: ' + response.message, 'error');
                        finishProcessing();
                    }
                },
                error: function(xhr, status, error) {
                    logMessage('AJAX Error: ' + error, 'error');
                    finishProcessing();
                }
            });
        }

        function processBatchFiles(files, formData, callback) {
            var fileIndex = 0;

            function processNextFile() {
                if (fileIndex >= files.length) {
                    callback();
                    return;
                }

                var file = files[fileIndex];
                $('#current-file').text('Processing: ' + file.original_filename);

                $.ajax({
                    url: json_strings.uri.base + 'includes/ajax.process.php?do=thumbnails_regenerate_process',
                    type: 'POST',
                    data: {
                        file_id: file.id,
                        width: formData.width,
                        height: formData.height,
                        csrf_token: document.getElementById('csrf_token').value
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.status === 'success') {
                            processedFiles++;
                            logMessage('✓ ' + response.data.filename, 'success');
                        } else {
                            failedFiles++;
                            logMessage('✗ ' + (response.data ? response.data.filename : 'Unknown file') + ': ' + response.message, 'error');
                        }
                        
                        updateProgress();
                        fileIndex++;
                        
                        // Small delay between files to prevent overwhelming the server
                        setTimeout(processNextFile, 100);
                    },
                    error: function(xhr, status, error) {
                        failedFiles++;
                        logMessage('✗ ' + file.original_filename + ': AJAX Error - ' + error, 'error');
                        updateProgress();
                        fileIndex++;
                        setTimeout(processNextFile, 100);
                    }
                });
            }

            processNextFile();
        }

        function updateProgress() {
            var totalProcessed = processedFiles + failedFiles;
            var percentage = totalFiles > 0 ? Math.round((totalProcessed / totalFiles) * 100) : 0;
            
            $('#progress-bar').css('width', percentage + '%');
            $('#progress-text').text(percentage + '%');
            $('#processed-count').text(processedFiles);
            $('#failed-count').text(failedFiles);
        }

        function logMessage(message, type) {
            var className = type === 'error' ? 'text-danger' : (type === 'success' ? 'text-success' : '');
            var timestamp = new Date().toLocaleTimeString();
            $('#process-log').append('<div class="' + className + '">[' + timestamp + '] ' + message + '</div>');
            $('#process-log').scrollTop($('#process-log')[0].scrollHeight);
        }

        function finishProcessing() {
            isProcessing = false;
            
            var regenerateButton = $('#regenerate-btn');
            var form = $('#thumbnails-form');
            
            $('#current-file').text('');
            regenerateButton.prop('disabled', false).html('<i class="fa fa-refresh"></i> ' + json_strings.thumbnails_regenerate.start_regeneration);
            form.find('input, button').prop('disabled', false);

            var message = json_strings.thumbnails_regenerate.process_completed
                .replace('{processed}', processedFiles)
                .replace('{failed}', failedFiles)
                .replace('{total}', processedFiles + failedFiles);

            logMessage(message, 'info');

            // Show completion message
            if (failedFiles > 0) {
                Swal.fire({
                    title: json_strings.thumbnails_regenerate.process_complete_errors_title,
                    text: json_strings.thumbnails_regenerate.completed_with_errors
                        .replace('{processed}', processedFiles)
                        .replace('{failed}', failedFiles),
                    icon: 'warning',
                    showConfirmButton: true,
                    confirmButtonText: json_strings.translations.ok,
                    buttonsStyling: false,
                    customClass: {
                        confirmButton: 'btn btn-primary'
                    },
                    showClass: {
                        popup: 'animate__animated animate__fadeIn'
                    },
                    hideClass: {
                        popup: 'animate__animated animate__fadeOut'
                    }
                });
            } else {
                Swal.fire({
                    title: json_strings.thumbnails_regenerate.process_complete_title,
                    text: json_strings.thumbnails_regenerate.completed_successfully
                        .replace('{processed}', processedFiles),
                    icon: 'success',
                    showConfirmButton: true,
                    confirmButtonText: json_strings.translations.ok,
                    buttonsStyling: false,
                    customClass: {
                        confirmButton: 'btn btn-primary'
                    },
                    showClass: {
                        popup: 'animate__animated animate__fadeIn'
                    },
                    hideClass: {
                        popup: 'animate__animated animate__fadeOut'
                    }
                });
            }
        }
    };
})();
(function () {
    'use strict';

    admin.pages.updates = function () {
        function initUpdates() {
            console.log('Updates page loaded');
            console.log('json_strings available:', typeof json_strings !== 'undefined');
            if (typeof json_strings !== 'undefined') {
                console.log('Base URI:', json_strings.uri.base);
            }
            var requirementsCheck = document.getElementById('requirements-check');
            var requirementsCard = document.getElementById('requirements-card');
            var progressCard = document.getElementById('update-progress-card');
            var startUpdateButton = document.getElementById('start-update');
            var progressBar = document.querySelector('.progress-bar');
            var progressStatus = document.getElementById('progress-status');
            var updateError = document.getElementById('update-error');
            var errorMessage = document.getElementById('error-message');
            var updateSuccess = document.getElementById('update-success');
            var requirementsList = document.querySelector('.requirements-list');

            var updateSteps = ['download', 'backup', 'extract', 'finalize'];
            var currentStep = 0;

            // Check system requirements on page load
            checkRequirements();

            // Handle update button click
            if (startUpdateButton) {
                startUpdateButton.addEventListener('click', startUpdate);
            }

            function checkRequirements() {
                console.log('Starting requirements check');
                console.log('URL:', json_strings.uri.base + 'process.php?do=check_update_requirements');

                // Prepare request data with CSRF token for POST request
                var formData = new FormData();
                var csrfToken = document.getElementById('csrf_token');
                if (csrfToken) {
                    formData.append('csrf_token', csrfToken.value);
                }

                fetch(json_strings.uri.base + 'process.php?do=check_update_requirements', {
                    method: 'POST',
                    body: formData
                })
                    .then(response => response.json())
                    .then(data => {
                        // Hide loading indicator
                        requirementsCheck.querySelector('.text-center').style.display = 'none';
                        requirementsList.style.display = 'block';
                        requirementsList.innerHTML = '';

                        // Display requirements
                        if (data.requirements) {
                            data.requirements.forEach(function(req) {
                                var li = document.createElement('li');
                                li.className = req.status ? 'requirement-pass' : 'requirement-fail';
                                li.innerHTML = '<strong>' + req.name + '</strong>: ' + req.message;
                                requirementsList.appendChild(li);
                            });
                        }

                        // Enable update button if all requirements pass
                        if (data.can_update) {
                            startUpdateButton.disabled = false;
                            startUpdateButton.classList.remove('btn-secondary');
                            startUpdateButton.classList.add('btn-primary');
                        } else {
                            startUpdateButton.disabled = true;
                            startUpdateButton.innerHTML = '<i class="fa fa-times"></i> ' + json_strings.updates.cannot_update;
                        }
                    })
                    .catch(error => {
                        console.error('Error checking requirements:', error);
                        console.log('Full error details:', error);
                        requirementsCheck.querySelector('.text-center').style.display = 'none';
                        requirementsList.style.display = 'block';
                        requirementsList.innerHTML = '<li class="requirement-fail">' + json_strings.updates.error_checking_requirements + '</li>';
                    });
            }

            function startUpdate() {
                // Disable button
                startUpdateButton.disabled = true;

                // Hide requirements card and show progress card
                requirementsCard.style.display = 'none';
                progressCard.style.display = 'block';
                updateError.style.display = 'none';
                updateSuccess.style.display = 'none';

                // Smooth scroll to top
                window.scrollTo({
                    top: 0,
                    behavior: 'smooth'
                });

                // Start with first step
                currentStep = 0;
                processUpdateStep();
            }

            function processUpdateStep() {
                if (currentStep >= updateSteps.length) {
                    // Update completed
                    showSuccess();
                    return;
                }

                var step = updateSteps[currentStep];
                updateStepIndicator(step, 'active');
                updateProgressBar((currentStep / updateSteps.length) * 100);

                var stepMessages = {
                    'download': json_strings.updates.downloading_update,
                    'backup': json_strings.updates.creating_backup,
                    'extract': json_strings.updates.installing_files,
                    'finalize': json_strings.updates.finalizing_update
                };

                progressStatus.textContent = stepMessages[step] || json_strings.updates.processing;

                // Prepare request data
                var formData = new FormData();
                formData.append('step', step);

                // Add CSRF token
                var csrfToken = document.getElementById('csrf_token');
                if (csrfToken) {
                    formData.append('csrf_token', csrfToken.value);
                }

                if (step === 'download' && typeof update_download_url !== 'undefined') {
                    formData.append('url', update_download_url);
                    // Add SHA256 hash if available
                    if (typeof update_data !== 'undefined' && update_data.sha256) {
                        formData.append('hash', update_data.sha256);
                    }
                }

                // Execute step
                fetch(json_strings.uri.base + 'process.php?do=perform_system_update', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        updateStepIndicator(step, 'complete');
                        currentStep++;

                        // Continue to next step
                        setTimeout(function() {
                            processUpdateStep();
                        }, 500);
                    } else {
                        // Error occurred - redirect with error message
                        console.error('Update step failed:', data);
                        console.log('Showing error:', data.message || json_strings.updates.update_failed);

                        var errorMsg = data.message || json_strings.updates.update_failed;
                        window.location.href = json_strings.uri.base + 'updates.php?error=' + encodeURIComponent(errorMsg);
                    }
                })
                .catch(error => {
                    console.error('Update error:', error);
                    var errorMsg = json_strings.updates.update_failed + ': ' + error.message;
                    window.location.href = json_strings.uri.base + 'updates.php?error=' + encodeURIComponent(errorMsg);
                });
            }

            function attemptRollback() {
                progressStatus.textContent = json_strings.updates.rolling_back;
                updateProgressBar(0);

                var formData = new FormData();
                formData.append('step', 'rollback');

                // Add CSRF token
                var csrfToken = document.getElementById('csrf_token');
                if (csrfToken) {
                    formData.append('csrf_token', csrfToken.value);
                }

                fetch(json_strings.uri.base + 'process.php?do=perform_system_update', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        errorMessage.innerHTML += '<br>' + json_strings.updates.rollback_successful;
                    } else {
                        errorMessage.innerHTML += '<br>' + json_strings.updates.rollback_failed;
                    }
                })
                .catch(error => {
                    console.error('Rollback error:', error);
                    errorMessage.innerHTML += '<br>' + json_strings.updates.rollback_failed;
                });
            }

            function updateStepIndicator(step, status) {
                var indicator = document.querySelector('.step-indicator[data-step="' + step + '"]');
                if (indicator) {
                    // Remove all status classes
                    indicator.classList.remove('step-active', 'step-complete', 'step-error');

                    // Add appropriate class
                    if (status === 'active') {
                        indicator.classList.add('step-active');
                    } else if (status === 'complete') {
                        indicator.classList.add('step-complete');
                    } else if (status === 'error') {
                        indicator.classList.add('step-error');
                    }
                }
            }

            function updateProgressBar(percent) {
                progressBar.style.width = percent + '%';
                progressBar.setAttribute('aria-valuenow', percent);
            }

            function showError(message) {
                progressCard.style.display = 'block';
                updateError.style.display = 'block';
                errorMessage.textContent = message;

                // Mark current step as error
                if (currentStep < updateSteps.length) {
                    updateStepIndicator(updateSteps[currentStep], 'error');
                }
            }

            function showSuccess() {
                progressBar.classList.remove('progress-bar-animated');
                progressBar.classList.add('bg-success');
                updateProgressBar(100);
                progressStatus.textContent = json_strings.updates.update_complete;
                updateSuccess.style.display = 'block';

                // Mark all steps as complete
                updateSteps.forEach(function(step) {
                    updateStepIndicator(step, 'complete');
                });

                // Redirect to dashboard after 5 seconds
                setTimeout(function() {
                    window.location.href = json_strings.uri.base + 'dashboard.php';
                }, 5000);
            }
        }

        // Check if DOM is already loaded or wait for it
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initUpdates);
        } else {
            initUpdates();
        }
    };
})();
(function () {
    'use strict';

    admin.pages.uploadForm = function () {

        $(document).ready(function(){
            var file_ids = [];
            var errors = 0;
            var successful = 0;

            // Send a keep alive action every 1 minute
            setInterval(function(){
                var timestamp = new Date().getTime()
                $.ajax({
                    type:	'GET',
                    cache:	false,
                    url:	json_strings.uri.base + 'includes/ajax-keep-alive.php',
                    data:	'timestamp='+timestamp,
                    success: function(result) {
                        var dummy = result;
                    }
                });
            },1000*60);

            var uploader = $('#uploader').pluploadQueue();

            $('#upload_form').on('submit', function(e) {
                if (uploader.files.length > 0) {
                    uploader.bind('StateChanged', function() {
                        if (uploader.files.length === (uploader.total.uploaded + uploader.total.failed)) {
                            var action = $('#upload_form').attr('action') + '?ids=' + file_ids.toString() + '&type=new';
                            $('#upload_form').attr('action', action);
                            if (successful > 0) {
                                if (errors == 0) {
                                    window.location = action;
                                } else {
                                    $(`
                                        <div class="alert alert-info">`+json_strings.translations.upload_form.some_files_had_errors+`</div>
                                        <a class="btn btn-wide btn-primary" href="`+action+`">`+json_strings.translations.upload_form.continue_to_editor+`</a>
                                    `).insertBefore( "#upload_form" );
                                }
                                return;
                            }
                            // $('#upload_form')[0].submit();
                        }
                    });

                    uploader.start();

                    $("#btn-submit").hide();
                    $(".message_uploading").fadeIn();

                    uploader.bind('Error', function(uploader, error) {
                        var obj = JSON.parse(error.response);
                        $(
                            `<div class="alert alert-danger">`+obj.error.filename+`: `+obj.error.message+`</div>`
                        ).insertBefore( "#upload_form" );
                        //console.log(obj);
                    });
        
                    uploader.bind('FileUploaded', function (uploader, file, info) {
                        var obj = JSON.parse(info.response);
                        file_ids.push(obj.info.id);
                        successful++;
                    });

                    return false;
                } else {
                    alert(json_strings.translations.upload_form.no_files);
                }

                return false;
            });

            window.onbeforeunload = function (e) {
                var e = e || window.event;

                console.log('state? ' + uploader.state);

                // if uploading
                if(uploader.state === 2) {
                    //IE & Firefox
                    if (e) {
                        e.returnValue = json_strings.translations.upload_form.leave_confirm;
                    }

                    // For Safari
                    return json_strings.translations.upload_form.leave_confirm;
                }

            };

        });
    };
})();
(function () {
    'use strict';

    admin.pages.userForm = function () {
        var form = document.getElementById('user_form');
        var formChanged = false;
        var formSubmitting = false;
        var initialFormData = form ? new FormData(form) : null;

        $(document).ready(function(){
            // Display or not the limit accounts selector
            $('#level').on('change', function(e) {
                var el = document.getElementById('limit_upload_to_container');
                if (typeof(el) != 'undefined' && el != null) {
                    if ($(this).val() == '7') {
                        $('#limit_upload_to_container').slideDown();
                    } else {
                        $('#limit_upload_to_container').slideUp();
                    }
                }
            });

            var form_type = $("#user_form").data('form-type');

            var validator = $("#user_form").validate({
                rules: {
                    name: {
                        required: true
                    },
                    username: {
                        required: true,
                        minlength: json_strings.character_limits.user_min,
                        maxlength: json_strings.character_limits.user_max,
                        alphanumericUsername: true
                    },
                    email: {
                        required: true,
                        email: true
                    },
                    level: {
                        required: true
                    },
                    max_file_size: {
                        required: true,
                        digits: true
                    },
                    password: {
                        required: {
                            param: true,
                            depends: function(element) {
                                if (form_type == 'new_user') {
                                    return true;
                                }
                                if (form_type == 'edit_user' || form_type == 'edit_user_self') {
                                    if ($.trim($("#password").val()).length > 0) {
                                        return true;
                                    }
                                }
                                return false;
                            }
                        },
                        minlength: json_strings.character_limits.password_min,
                        maxlength: json_strings.character_limits.password_max,
                        passwordValidCharacters: true
                    }
                },
                messages: {
                    name: {
                        required: json_strings.validation.no_name
                    },
                    username: {
                        required: json_strings.validation.no_user,
                        minlength: json_strings.validation.length_user,
                        maxlength: json_strings.validation.length_user,
                        alphanumericUsername: json_strings.validation.alpha_user
                    },
                    email: {
                        required: json_strings.validation.no_email,
                        email: json_strings.validation.invalid_email
                    },
                    level: {
                        required: json_strings.validation.no_role
                    },
                    max_file_size: {
                        digits: json_strings.validation.file_size
                    },
                    password: {
                        required: json_strings.validation.no_pass,
                        minlength: json_strings.validation.length_pass,
                        maxlength: json_strings.validation.length_pass,
                        passwordValidCharacters: json_strings.validation.alpha_pass
                    }
                },
                errorPlacement: function(error, element) {
                    error.appendTo(element.parent('div'));
                }
            });
        });

        // Track form changes for unsaved changes warning
        if (form) {
            // Track changes to all form inputs
            form.addEventListener('change', function() {
                formChanged = true;
            });

            // Also track text input changes
            var textInputs = form.querySelectorAll('input[type="text"], input[type="email"], input[type="password"], input[type="number"], input[type="checkbox"], textarea, select');
            textInputs.forEach(function(input) {
                input.addEventListener('input', function() {
                    // Check if form has actually changed from initial state
                    var currentFormData = new FormData(form);
                    formChanged = !areFormDataEqual(initialFormData, currentFormData);
                });
            });

            // Set flag when form is being submitted
            form.addEventListener('submit', function() {
                formSubmitting = true;
            });

            // Helper function to compare FormData objects
            function areFormDataEqual(formData1, formData2) {
                if (!formData1 || !formData2) return false;

                var entries1 = Array.from(formData1.entries());
                var entries2 = Array.from(formData2.entries());

                if (entries1.length !== entries2.length) {
                    return false;
                }

                for (var i = 0; i < entries1.length; i++) {
                    if (entries1[i][0] !== entries2[i][0] || entries1[i][1] !== entries2[i][1]) {
                        return false;
                    }
                }

                return true;
            }

            // Warn user before leaving if there are unsaved changes
            window.addEventListener('beforeunload', function(e) {
                if (formChanged && !formSubmitting) {
                    var confirmationMessage = 'You have unsaved changes. Are you sure you want to leave this page?';
                    e.returnValue = confirmationMessage;
                    return confirmationMessage;
                }
            });

            // Handle navigation away from the page (for links within the application)
            document.addEventListener('click', function(e) {
                // Check if it's a link that would navigate away
                var target = e.target.closest('a');
                if (target && target.href && !target.href.startsWith('#') && !target.classList.contains('delete-confirm')) {
                    if (formChanged && !formSubmitting) {
                        e.preventDefault();
                        var targetUrl = target.href;

                        Swal.fire({
                            title: 'Unsaved Changes',
                            text: 'You have unsaved changes. Are you sure you want to leave this page?',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#d33',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Yes, leave page',
                            cancelButtonText: 'Stay on page',
                            showClass: {
                                popup: 'animate__animated animate__fadeIn'
                            },
                            hideClass: {
                                popup: 'animate__animated animate__fadeOut'
                            }
                        }).then(function(result) {
                            if (result.isConfirmed) {
                                formSubmitting = true; // Prevent the beforeunload warning
                                window.location.href = targetUrl;
                            }
                        });
                    }
                }
            });
        }
    };
})();
/**
 * Apply bulk actions
 */
(function () {
    'use strict';

    admin.parts.bulkActions = function () {

        $(document).ready(function(e) {
            $(".batch_actions").on('submit', function(e) {
                var checks = $("input[name='batch[]']").serializeArray();
                var action = $('#action').val();
                if (action != 'none') {
                        // Generic actions
                        if (action == 'delete') {
                            if (checks.length == 0) {
                                alert(json_strings.translations.select_one_or_more);
                                return false;
                            }
                            else {
                                var _formatted = sprintf(json_strings.translations.confirm_delete, checks.length);
                                if (!confirm(_formatted)) {
                                    e.preventDefault();
                                }
                            }
                        }

                        // Activities log actions
                        if (action == 'log_clear') {
                            var msg = json_strings.translations.confirm_delete_log;
                            if (!confirm(msg)) {
                                e.preventDefault();
                            }
                        }

                        if (action == 'log_download') {
                            e.preventDefault();

                            let modal_content = `<p class="loading-icon">
                                <img src="`+json_strings.uri.assets_img+`/loading.svg" alt="Loading" /></p>
                                <p class="lead">`+json_strings.translations.download_wait+`</p>
                            `;

                            Cookies.set('log_download_started', 0, { expires: 100 });
                            setTimeout(check_log_download_cookie, 1000);

                            Swal.fire({
                                title: '',
                                html: modal_content,
                                showCloseButton: false,
                                showCancelButton: false,
                                showConfirmButton: false,
                                showClass: {
                                    popup: 'animate__animated animated__fast animate__fadeInUp'
                                },
                                hideClass: {
                                    popup: 'animate__animated animated__fast animate__fadeOutDown'
                                },
                                didOpen: function () {
                                    var url = json_strings.uri.base+'includes/actions.log.export.php?format=csv';
                                    let iframe = document.createElement('iframe');
                                    let swalcontainer = document.getElementById('swal2-html-container')
                                    swalcontainer.appendChild(iframe);
                                    iframe.setAttribute('src', url);
                                }
                            }).then((result) => {
                            });
                        }

                        if (action == 'cron_log_download') {
                            e.preventDefault();

                            let modal_content = `<p class="loading-icon">
                                <img src="`+json_strings.uri.assets_img+`/loading.svg" alt="Loading" /></p>
                                <p class="lead">`+json_strings.translations.download_wait+`</p>
                            `;

                            Cookies.set('log_download_started', 0, { expires: 100 });
                            setTimeout(check_log_download_cookie, 1000);

                            Swal.fire({
                                title: '',
                                html: modal_content,
                                showCloseButton: false,
                                showCancelButton: false,
                                showConfirmButton: false,
                                showClass: {
                                    popup: 'animate__animated animated__fast animate__fadeInUp'
                                },
                                hideClass: {
                                    popup: 'animate__animated animated__fast animate__fadeOutDown'
                                },
                                didOpen: function () {
                                    var url = json_strings.uri.base+'includes/cron.log.export.php?format=csv';
                                    let iframe = document.createElement('iframe');
                                    let swalcontainer = document.getElementById('swal2-html-container')
                                    swalcontainer.appendChild(iframe);
                                    iframe.setAttribute('src', url);
                                }
                            }).then((result) => {
                            });
                        }

                        // Manage files actions
                        if (action == 'unassign') {
                            var _formatted = sprintf(json_strings.translations.confirm_unassign, checks.length);
                            if (!confirm(_formatted)) {
                                e.preventDefault();
                            }
                        }

                        // Download multiple files as zip
                        if (action == 'zip') {
                            e.preventDefault();
                            var checkboxes = $("input[name='batch[]']:checked").serializeArray();
                            if (checkboxes.length > 0) {
                                let modal_content = `<p class="loading-icon"><img src="`+json_strings.uri.assets_img+`/loading.svg" alt="Loading" /></p>
                                    <p class="lead">`+json_strings.translations.download_wait+`</p>
                                    <p class="">`+json_strings.translations.download_long_wait+`</p>
                                `;

                                Cookies.set('download_started', 0, { expires: 100 });
                                setTimeout(check_download_cookie, 1000);

                                Swal.fire({
                                    title: '',
                                    html: modal_content,
                                    showCloseButton: false,
                                    showCancelButton: false,
                                    showConfirmButton: false,
                                    showClass: {
                                        popup: 'animate__animated animated__fast animate__fadeInUp'
                                    },
                                    hideClass: {
                                        popup: 'animate__animated animated__fast animate__fadeOutDown'
                                    },
                                    didOpen: function () {
                                        $.ajax({
                                            method: 'GET',
                                            url: json_strings.uri.base + 'process.php',
                                            data: { do:"return_files_ids", files:checkboxes }
                                        }).done( function(rsp) {
                                            var url = json_strings.uri.base + 'process.php?do=download_zip&files=' + rsp;
                                            let iframe = document.createElement('iframe');
                                            let swalcontainer = document.getElementById('swal2-html-container')
                                            swalcontainer.appendChild(iframe);
                                            iframe.setAttribute('src', url);
                                        });
                                    }
                                }).then((result) => {
                                });
                            }
                        }
                }
                else {
                    return false;
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.downloadCookieHandler = function () {
        $(document).ready(function() {
            /**
             * CLOSE THE ZIP DOWNLOAD MODAL
             * Solution to close the modal. Suggested by remez, based on
             * https://stackoverflow.com/questions/29532788/how-to-display-a-loading-animation-while-file-is-generated-for-download
             */
            var downloadTimeout;
            window.check_download_cookie = function() {
                if (Cookies.get("download_started") == 1) {
                    Cookies.set("download_started", "false", { expires: 100 });
                    Swal.close();
                } else {
                    downloadTimeout = setTimeout(check_download_cookie, 1000);
                }
            };

            // Close the log CSV download modal
            var logdownloadTimeout;
            window.check_log_download_cookie = function() {
                if (Cookies.get("log_download_started") == 1) {
                    Cookies.set("log_download_started", "false", { expires: 100 });
                    Swal.close();
                } else {
                    logdownloadTimeout = setTimeout(check_log_download_cookie, 1000);
                }
            };
        });
    }
})();
(function () {
    'use strict';

    admin.parts.filePreviewModal = function () {

        $(document).ready(function(e) {
            // Append modal
            var modal_layout = `<div id="preview_modal" class="modal fade" tabindex="-1" role="dialog">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title"></h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                        </div>
                        <div class="modal-footer">
                        </div>
                    </div>
                </div>
            </div>`;
            $('body').append(modal_layout);

            // Button trigger
            $('.get-preview').on('click', function(e) {
                e.preventDefault();
                var url = $(this).data("url"); 
                var content = '';

                $.ajax({
                    method: "GET",
                    url: url,
                    cache: false,
                }).done(function(response) {
                    var obj = JSON.parse(response);
                    switch (obj.type) {
                        case 'video':
                            content = `
                                <div class="embed-responsive embed-responsive-16by9">
                                    <video controls class="w-100">
                                        <source src="`+obj.file_url+`" format="`+obj.mime_type+`">
                                    </video>
                                </div>`;
                            break;
                        case 'audio':
                            content = `
                                <audio controls>
                                    <source src="`+obj.file_url+`" format="`+obj.mime_type+`">
                                </audio>`;
                            break;
                        case 'pdf':
                            content = `
                                <div class="embed-responsive embed-responsive-16by9">
                                    <iframe src="`+obj.file_url+`"></iframe>
                                </div>
                            `;
                            break;
                        case 'image':
                            content = `<img src="`+obj.file_url+`" class="img-responsive">`
                            break;
                        }
                    $('.modal-header h5').html(obj.name);
                    $('.modal-body').html(content);
                    // show modal
                    $('#preview_modal').modal('show');
                }).fail(function(response) {
                    alert(json_strings.translations.preview_failed);
                }).always(function() {
                });    
            });

            // Remove content when closing modal
            $('#preview_modal').on('hidden.bs.modal', function (e) {
                $('.modal-body').html('');
            })
        });
    };
})();

(function () {
    'use strict';

    admin.parts.foldersAdmin = function () {
        let btn_create = document.getElementById('btn_header_folder_create');
        if (btn_create != null && btn_create != 'undefined') {
            btn_create.addEventListener('click', function(e) {
                createFolderModal();
            });
        }

        async function createFolderModal()
        {
            const { value: folder_name } = await Swal.fire({
                title: btn_create.dataset.modalTitle,
                input: 'text',
                inputLabel: btn_create.dataset.modalLabel,
                inputValue: null,
                showCancelButton: true,
                inputAttributes: {
                    maxlength: 100,
                    autocapitalize: 'off',
                    autocorrect: 'off'
                },
                inputValidator: (value) => {
                    if (!value) {
                        return btn_create.dataset.modalTitleInvalid
                    }
                }
            })

            if (folder_name) {
                var data = new FormData();
                data.append('csrf_token', document.getElementById('csrf_token').value);
                data.append('folder_name', folder_name);
                data.append('folder_parent', btn_create.dataset.parent);
                
                axios.post(btn_create.dataset.processUrl, data)
                .then(function (response) {
                    const folder_url = btn_create.dataset.folderUrl;
                    const folder_template = document.getElementById('new_folder').content.cloneNode(true);
                    let new_folder = folder_template.querySelectorAll('div')[0];
                    new_folder.classList.add('new_flash');
                    new_folder.dataset.folderId = response.data.data.id;
                    new_folder.querySelector('a').href = folder_url.replace('{folder_id}', response.data.data.id);
                    new_folder.querySelector('a span').innerText = folder_name;

                    const folders_nav = document.getElementById('folders_nav');
                    folders_nav.appendChild(new_folder);
                    addFolderDragEvents();
                })
                .catch(function (error) {
                    console.log(error);
                    // new Toast(error.response.data.error, Toast.TYPE_ERROR, Toast.TIME_NORMAL);
                });
            }
        }

        // Drag and drop events
        addFolderDragEvents();

        function foldersDropTargetMake(except)
        {
            let draggable_folders = document.querySelectorAll('.folder_destination');
            for (let i = 0; i < draggable_folders.length; i++) {
                if (except != draggable_folders[i]) {
                    draggable_folders[i].classList.add("drop_target_active");
                }
                else {
                    draggable_folders[i].classList.add("drop_target_is_self");
                }
            }
        }

        function foldersDropTargetRemove()
        {
            let draggable_folders = document.querySelectorAll('.folder_destination');
            for (let i = 0; i < draggable_folders.length; i++) {
                draggable_folders[i].classList.remove("drop_target_active");
                draggable_folders[i].classList.remove("drop_target_is_self");
                draggable_folders[i].classList.remove("drop_forbidden");
            }
        }

        function addFolderDragEvents() {
            const folders_nav = document.getElementById('folders_nav');
            if (typeof(folders_nav) != 'undefined' && folders_nav != null)
            {
                let draggable_folders = document.querySelectorAll('.folder_destination');
                let draggable_files = document.querySelectorAll('.file_draggable');

                // Folders
                for (let i = 0; i < draggable_folders.length; i++) {
                    if (draggable_folders !== null) {
                        // Start
                        draggable_folders[i].addEventListener("dragstart", function(e) {
                            // e.preventDefault();
                            draggable_folders[i].classList.add("dragging");
                            foldersDropTargetMake(draggable_folders[i]);
                        }, true);

                        // Stop
                        draggable_folders[i].addEventListener("dragend", function(e) {
                            // e.preventDefault();
                            draggable_folders[i].classList.remove("dragging");
                            foldersDropTargetRemove();
                        }, true);
                    }
                }

                // Files
                if (draggable_files !== null) {
                    for (let i = 0; i < draggable_files.length; i++) {
                        // Start
                        draggable_files[i].addEventListener("dragstart", function(e) {
                            // e.preventDefault();
                            draggable_files[i].classList.add("dragging");
                            foldersDropTargetMake();
                        }, true);
    
                        // Stop
                        draggable_files[i].addEventListener("dragend", function(e) {
                            // e.preventDefault();
                            draggable_files[i].classList.remove("dragging");
                            foldersDropTargetRemove();
                        }, true);
                    }
                }

                let draggable_destinations = document.querySelectorAll('.folder_destination');
                if (draggable_destinations !== null) {
                    for (let i = 0; i < draggable_destinations.length; i++) {
                        // Drag another item over
                        draggable_destinations[i].addEventListener("dragover", function(e) {
                            e.preventDefault();
                            const dragged_item = document.querySelector('.dragging');
                            if (e.currentTarget.dataset.folderId == dragged_item.dataset.folderId) {
                                draggable_destinations[i].classList.add("drop_forbidden");
                            } else {
                                draggable_destinations[i].classList.add("drop_ready");
                            }
                        }, true);

                        // Stop drag over
                        draggable_destinations[i].addEventListener("dragleave", function(e) {
                            // e.preventDefault();
                            draggable_destinations[i].classList.remove("drop_forbidden");
                            draggable_destinations[i].classList.remove("drop_ready");
                        }, true);

                        // Drop an element inside
                        draggable_destinations[i].addEventListener("drop", function(e) {
                            let url;
                            e.preventDefault();
                            draggable_destinations[i].classList.remove("drop_forbidden");
                            draggable_destinations[i].classList.remove("drop_ready");
                            draggable_destinations[i].classList.add("dropped");
                            
                            const dragged_item = document.querySelector('.dragging');
                            const destiny = e.currentTarget;

                            const dragged_type = dragged_item.dataset.draggableType;
                            if (e.currentTarget.dataset.folderId == dragged_item.dataset.folderId) {
                                return;
                            }

                            var data = new FormData();
                            data.append('csrf_token', document.getElementById('csrf_token').value);
                            data.append('new_parent_id', destiny.dataset.folderId);
            
                            switch (dragged_type) {
                                case 'folder':
                                    url = document.getElementById('folder_context_menu__links').dataset.urlFolderOndrop;
                                    data.append('folder_id', dragged_item.dataset.folderId);
                                    dragged_item.classList.add('d-none');
                                    axios.post(url, data)
                                    .then(function (response) {
                                        dragged_item.remove();
                                    })
                                    .catch(function (error) {
                                        dragged_item.classList.remove('d-none');
                                        // console.log(error);
                                        // new Toast(error.response.data.error, Toast.TYPE_ERROR, Toast.TIME_NORMAL);
                                    });
                                break;
                                case 'file':
                                    url = document.getElementById('folder_context_menu__links').dataset.urlFileOndrop;
                                    data.append('file_id', dragged_item.dataset.fileId);
                                    dragged_item.classList.add('d-none');
                                    axios.post(url, data)
                                    .then(function (response) {
                                        dragged_item.remove();
                                    })
                                    .catch(function (error) {
                                        dragged_item.classList.remove('d-none');
                                    });
                                break;
                            }
                        }, true);
                    }
                }

                // Context menu
                let folders_items = document.querySelectorAll('.folder');
                
                for (let i = 0; i < folders_items.length; i++) {
                    const can_edit = folders_items[i].dataset.canEdit;
                    const can_delete = folders_items[i].dataset.canDelete;

                    let menu_items = [];

                    menu_items.push({
                        label: 'Navigate',
                        iconClass: 'fa fa-arrow-right',
                        callback: async () => {
                            folderNavigate(folders_items[i]);
                        }
                    });
                    menu_items.push('hr');

                    if (can_edit == 'true') {
                        menu_items.push({
                            label: 'Rename',
                            iconClass: 'fa fa-pencil',
                            callback: async () => {
                                folderRename(folders_items[i]);
                            }
                        });
                        menu_items.push('hr');
                    }

                    if (can_delete == 'true') {
                        menu_items.push({
                            label: 'Delete',
                            iconClass: 'fa fa-trash-o',
                            callback: async () => {
                                folderDelete(folders_items[i]);
                            }
                        });
                    }

                    new VanillaContextMenu({
                        scope: folders_items[i],
                        customThemeClass: 'context-menu-orange-theme',
                        customClass: 'custom-context-menu-cls',
                        menuItems: menu_items
                    });
                }
            }
        }

        document.addEventListener('click',function(e) {
            let clickedFolder = e.target.closest('.folder_open_context_menu');
            if (clickedFolder) {
                var evt = new MouseEvent('contextmenu', {
                    bubbles: true,
                    cancelable: true,
                    view: window,
                    buttons: 2,
                    clientX: clickedFolder.getBoundingClientRect().x,
                    clientY: clickedFolder.getBoundingClientRect().y
                });
                clickedFolder.dispatchEvent(evt);
            }
        });
        


        function folderNavigate(folder)
        {
            window.location = folder.querySelectorAll('a')[0].href;
        }

        // function folderShare(folder)
        // {
        //     const url = document.getElementById('folder_context_menu__links').dataset.urlShare;
        // }

        async function folderRename(folder)
        {
            const url = document.getElementById('folder_context_menu__links').dataset.urlRename;
            const name_container = folder.querySelectorAll('a span')[0];
            const previous_name = name_container.innerHTML;
            const { value: new_name } = await Swal.fire({
                title: null,
                input: 'text',
                inputValue: previous_name,
                showCancelButton: true,
                inputAttributes: {
                    maxlength: 100,
                    autocapitalize: 'off',
                    autocorrect: 'off'
                },
                inputValidator: (value) => {
                    if (!value) {
                        return 'Name is not valid'
                    }
                }
            })
                
            if (new_name) {
                var data = new FormData();
                data.append('csrf_token', document.getElementById('csrf_token').value);
                data.append('folder_id', folder.dataset.folderId);
                data.append('name', new_name);
                name_container.innerHTML = new_name;
                
                axios.post(url, data)
                .then(function (response) {
                    folder.dataset.name = new_name;
                })
                .catch(function (error) {
                    name_container.innerHTML = previous_name;
                    new Toast(error.response.data.error, Toast.TYPE_ERROR, Toast.TIME_NORMAL);
                });
            }
        }

        function folderDelete(folder)
        {
            const url = document.getElementById('folder_context_menu__links').dataset.urlDelete;
            const folder_name = folder.dataset.name;
            const html = `Delete folder ${folder_name} and all of its contents?`

            Swal.fire({
                title: 'Delete folder',
                html: html,
                showCloseButton: false,
                showCancelButton: true,
                showConfirmButton: true,
                focusCancel: true,
                cancelButtonText: 'Cancel',
                confirmButtonText: 'Delete',
                buttonsStyling: false,
                reverseButtons: true,
                customClass: {
                    cancelButton: 'btn btn-lg btn-secondary',
                    confirmButton: 'btn btn-lg btn-danger ms-4'
                },
                showClass: {
                    popup: 'animate__animated animated__fast animate__fadeInUp'
                },
                hideClass: {
                    popup: 'animate__animated animated__fast animate__fadeOutDown'
                }
            }).then((result) => {
                if (result.value) {
                    folder.classList.add('d-none');

                    var data = new FormData();
                    data.append('csrf_token', document.getElementById('csrf_token').value);
                    data.append('folder_id', folder.dataset.folderId);
                    
                    axios.post(url, data)
                    .then(function (response) {
                        folder.remove();
                    })
                    .catch(function (error) {
                        folder.classList.remove('d-none');
                        new Toast(error.response.data.error, Toast.TYPE_ERROR, Toast.TIME_NORMAL);
                    });
                }
            });

        }
    };
})();
(function () {
    'use strict';

    admin.parts.generatePassword = function () {
        $(document).ready(function () {
            var hdl = new Jen(true);
            hdl.hardening(true);

            $('.btn_generate_password').click(function(e) {
                var target = $(this).parents('.form-group').find('.attach_password_toggler');
                var min_chars = $(this).data('min');
                var max_chars = $(this).data('max');
                $(target).val( hdl.password( min_chars, max_chars ) );
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.jqueryValidationCustomMethods = function () {

        $(document).ready(function(){
            jQuery.validator.addMethod("alphanumericUsername", function(value, element) {
                return this.optional(element) || /^[\w.]+$/i.test(value);
            }, json_strings.validation.alpha_user);

            jQuery.validator.addMethod("passwordValidCharacters", function(value, element) {
                return this.optional(element) || /^[0-9a-zA-Z`!"?$%\^&*()_\-+={\[}\]:;@~#|<,>.'\/\\]+$/.test(value);
            }, json_strings.validation.alpha_user);
        });
    };
})();
(function () {
    'use strict';

    admin.parts.loadCKEditor = function () {

        $(document).ready(function() {
            if (document.querySelector('textarea.ckeditor') !== null) {
                // CKEditor
                ClassicEditor
                    .create( document.querySelector( '.ckeditor' ), {
                        removePlugins: [ 'Heading', 'CKFinder', 'Link' ],
                        toolbar: [ 'bold', 'italic', 'bulletedList', 'numberedList', 'blockQuote' ]
                    })
                    .then( editor => {
                        window.editor = editor;
                    } )
                    .catch( error => {
                        console.error( 'There was a problem initializing the editor.', error );
                    } );
            }
        });
    }
})();
(function () {
    'use strict';

    admin.parts.login2faInputs = function () {

        let request_button = document.getElementById('request_new_2fa_code');
        if (typeof(request_button) != 'undefined' && request_button != null) {
            request_button.addEventListener('click', function (event) {
                // request new
                const token = document.getElementsByName('token')[0].value;
                console.log(token);
            })

            // Countdown and enable button            
            const shouldEnableButton = setInterval(() => {
                const wait_until = request_button.dataset.time;
                var now = new Date();
                var dateUntil = new Date(wait_until);
                var diff = Math.ceil((dateUntil - now) / 1000);
                request_button.innerText = request_button.dataset.textWait.replace('{seconds}', diff);
                if (diff < 0) {
                    request_button.innerText = request_button.dataset.text;
                    request_button.removeAttribute('disabled');
                    clearInterval(shouldEnableButton);
                }
            }, 1000);
        }

        // Adapted from Jinul's answer at
        // https://stackoverflow.com/questions/41698357/how-to-partition-input-field-to-appear-as-separate-input-fields-on-screen
        function OTPInput() {
            const otp_input_1 = document.querySelector('#otp_inputs #n1');
            if (typeof(otp_input_1) != 'undefined' && otp_input_1 != null){
                otp_input_1.onpaste = pasteOTP;
            }

            const inputs = document.querySelectorAll('#otp_inputs > *[id]');
            for (let i = 0; i < inputs.length; i++) {
                inputs[i].addEventListener('input', function (event) {
                    handleOTPNumericInput(event, this);
                });

                inputs[i].addEventListener('keyup', function (event) {
                    handleOTPNumericInput(event, this);
                });

                inputs[i].addEventListener('paste', function (event) {
                    handleOTPNumericInput(event, this);
                });
            }
        }
        OTPInput();

        function handleOTPNumericInput(event, el) {
            if (event.key !== 'undefined' && event.key != 'Backspace') {
                if (!isNumeric(event.target.value)) {
                    el.value = '';
                    el.focus();
                    return;
                }
            }
            if (!event.target.value || event.target.value == '') {
                if (event.target.previousSibling.previousSibling) {
                    event.target.previousSibling.previousSibling.focus();
                }
            } else {
                if (event.target.nextSibling.nextSibling) {
                    event.target.nextSibling.nextSibling.focus();
                }
            }
        }

        function pasteOTP(event) {
            event.preventDefault();
            let elm = event.target;
            let pasteVal = event.clipboardData.getData('text').split("");
            if (pasteVal.length > 0) {
                while (elm) {
                    elm.value = pasteVal.shift();
                    elm = elm.nextSibling.nextSibling;
                }
                const last = document.getElementById('n6').focus();
            }
        }
    }
})();
(function () {
    'use strict';

    admin.parts.loginTabs = function () {
        $(document).ready(function() {
            // Initialize Bootstrap tabs if not already initialized
            if (typeof bootstrap !== 'undefined' && bootstrap.Tab) {
                // Bootstrap 5 style
                var triggerTabList = [].slice.call(document.querySelectorAll('#local-tab, #ldap-tab'));
                triggerTabList.forEach(function (triggerEl) {
                    var tabTrigger = new bootstrap.Tab(triggerEl);
                    
                    triggerEl.addEventListener('click', function (event) {
                        event.preventDefault();
                        tabTrigger.show();
                    });
                });
            } else {
                // Fallback for older Bootstrap or jQuery tabs
                $('#local-tab, #ldap-tab').on('click', function(e) {
                    e.preventDefault();
                    
                    // Remove active classes from all tabs and content
                    $('.nav-link').removeClass('active');
                    $('.tab-pane').removeClass('active show');
                    
                    // Add active class to clicked tab
                    $(this).addClass('active');
                    
                    // Show corresponding content
                    var target = $(this).data('bs-target') || $(this).attr('href');
                    $(target).addClass('active show');
                    
                    // Clear any previous form data and focus on the appropriate field
                    if (target === '#local') {
                        $('#username').focus();
                        $('#ldap_email, #ldap_password').val('');
                    } else if (target === '#ldap') {
                        $('#ldap_email').focus();
                        $('#username, #password').val('');
                    }
                });
            }
            
            // Handle URL hash for direct tab linking
            if (window.location.hash === '#ldap') {
                $('#ldap-tab').click();
            }
            
            // Clear form fields when switching tabs
            $('button[data-bs-toggle="tab"]').on('shown.bs.tab', function (e) {
                var target = $(e.target).data('bs-target');
                
                if (target === '#local') {
                    $('#ldap_email, #ldap_password').val('');
                    setTimeout(function() {
                        $('#username').focus();
                    }, 150);
                } else if (target === '#ldap') {
                    $('#username, #password').val('');
                    setTimeout(function() {
                        $('#ldap_email').focus();
                    }, 150);
                }
                
                // Clear any error messages
                $('.ajax_response').removeClass('alert-danger alert-success').html('').hide();
            });
            
            // Update browser history when switching tabs
            $('button[data-bs-toggle="tab"]').on('click', function (e) {
                var target = $(e.target).data('bs-target');
                if (target === '#ldap') {
                    history.replaceState(null, null, '#ldap');
                } else {
                    history.replaceState(null, null, window.location.pathname);
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.main = function () {

        $(document).ready(function() {
            // Add asterisks to labels for required fields
            $('.form-control.required, .form-select[required], input.required').each(function() {
                var $field = $(this);
                var $formGroup = $field.closest('.form-group');

                if ($formGroup.length) {
                    var $label = $formGroup.find('label').first();

                    // Check if asterisk already exists to avoid duplicates
                    if ($label.length && !$label.find('.required-indicator').length) {
                        $label.append('<span class="required-indicator" aria-label="required">*</span>');
                    }
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.mainMenuSidebar = function () {

        $(document).ready(function() {
            window.adjust_main_menu = function() {
                var window_width = jQuery(window).width();
                if ( window_width < 769 ) {
                    $('.main_menu .active .dropdown_content').hide();
                    $('.main_menu li').removeClass('active');
            
                    if ( !$('body').hasClass('menu_contracted') ) {
                        $('body').addClass('menu_contracted');
                    }
                }
            }

            adjust_main_menu();

            $('.main_menu > li.has_dropdown .nav_top_level').click(function(e) {
                e.preventDefault();

                var parent = $(this).parents('.has_dropdown');
                if ( $(parent).hasClass('active') ) {
                    $(parent).removeClass('active');
                    $(parent).find('.dropdown_content').stop().slideUp();
                }
                else {
                    if ( $('body').hasClass('menu_contracted') ) {
                        $('.main_menu li').removeClass('active');
                        $('.main_menu').find('.dropdown_content').stop().slideUp(100);
                    }
                    $(parent).addClass('active');
                    $(parent).find('.dropdown_content').stop().slideDown();
                }
            });

            $('.toggle_main_menu').click(function(e) {
                e.preventDefault();

                var window_width = jQuery(window).width();
                if ( window_width > 768 ) {
                    $('body').toggleClass('menu_contracted');
                    if ( $('body').hasClass('menu_contracted') ) {
                        Cookies.set("menu_contracted", 'true', { expires: 365 } );
                        $('.main_menu li').removeClass('active');
                        $('.main_menu').find('.dropdown_content').stop().hide();
                    }
                    else {
                        Cookies.set("menu_contracted", 'false', { expires: 365 } );
                        $('.current_nav').addClass('active');
                        $('.current_nav').find('.dropdown_content').stop().show();
                    }
                }
                else {
                    $('body').toggleClass('menu_hidden');
                    $('.main_menu li').removeClass('active');

                    if ( $('body').hasClass('menu_hidden') ) {
                        //Cookies.set("menu_hidden", 'true', { expires: 365 } );
                        $('.main_menu').find('.dropdown_content').stop().hide();
                    }
                    else {
                        //Cookies.set("menu_hidden", 'false', { expires: 365 } );
                    }
                }
            });
        });

        jQuery(window).resize(function($) {
            adjust_main_menu();
        });
    }
})();
(function () {
    'use strict';

    admin.parts.misc = function () {

        $(document).ready(function() {
            // Focus the first input on the page. Generally, it's the search box
            $('input:first').focus();

            // Generic confirm alert
            $('.confirm_generic').on('click', function(e) {
                if (!confirm(json_strings.translations.confirm_generic)) {
                    e.preventDefault();
                }
            });
    
            // Dismiss messages
            $('.message .close').on('click', function () {
                $(this).closest('.message').transition('fade');
            });

            // Common for all tables and card lists
            $("#select_all").click(function(){
                var status = $(this).prop("checked");
                /** Uncheck all first in case you used pagination */
                $("input[type=checkbox].batch_checkbox").prop("checked",false);
                $("input[type=checkbox].batch_checkbox:visible").prop("checked",status);
            });

            // Loose focus after clicking buttons
            $('button').on('click', function() {
                $(this).blur();
            });

            $('.copy_text').on('click', function(e) {
                let target_id = $(this).data('target');
                let target = document.getElementById(target_id);
                var element_type = target.tagName.toLowerCase();
                switch (element_type) {
                    case 'input':
                        copyTextToClipboard(target.value);
                    break;
                    default:
                        copyTextToClipboard(target.innerHTML);
                    break;
                }
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.passwordVisibilityToggle = function () {
        let password_inputs = document.querySelectorAll('.attach_password_toggler')
        password_inputs.forEach(element => {
            var id;
            if (element.hasAttribute('id')) {
                id = element.id;
            } else {
                let random = (Math.random() + 1).toString(36).substring(7);
                id = 'el_pass_'+random;
                element.setAttribute('id', id);
            }

            let button = document.createElement('button');
            button.classList.add('btn', 'btn-sm', 'input-group-btn', 'password_toggler');
            button.setAttribute('type', 'button');
            button.dataset.targetId = id;
            button.dataset.status = 'hidden';

            let icon = document.createElement('i');
            icon.classList.add('fa', 'fa-eye');
            button.appendChild(icon);

            var copy = element.cloneNode();
            var wrapper = document.createElement('div');
            wrapper.classList.add('input-group');
            wrapper.appendChild(copy);
            wrapper.appendChild(button);

            element.insertAdjacentElement("afterend", wrapper);
            element.remove();

            button.addEventListener( 'click', function ( event ) {
                let id = event.target.dataset.targetId;
                let target = document.getElementById(id);
                let me = event.target;
                let icon = this.querySelector('i');
                let type;
                switch (me.dataset.status) {
                    case 'hidden':
                        icon.classList.remove('fa-eye');
                        icon.classList.add('fa-eye-slash');
                        me.dataset.status = 'visible';
                        type = 'text';
                    break;
                    case 'visible':
                        icon.classList.remove('fa-eye-slash');
                        icon.classList.add('fa-eye');
                        me.dataset.status = 'hidden';
                        type = 'password';
                    break;
                }

                let new_input = document.createElement('input');
                new_input.setAttribute('type', type);
                new_input.setAttribute('id', id);
                new_input.setAttribute('name', target.name);
                if (target.hasAttribute('max-length')) {
                    new_input.maxLength = target.maxLength;
                }
                new_input.className = target.className;
                new_input.value = target.value;
                target.insertAdjacentElement("beforebegin", new_input);
                target.remove();
            }, true);
        });
    };
})();
(function () {
    'use strict';

    admin.parts.publicLinksPopup = function () {
        $(document).ready(function () {
            /**
             * Modal: show a public file's URL
             */
            $('body').on('click', '.public_link', function (e) {
                var type = $(this).data('type');
                var file_title = $(this).data('title');
                var public_url = $(this).data('public-url');

                if (type == 'group') {
                    var note_text = json_strings.translations.public_group_note;
                } else if (type == 'file') {
                    var note_text = json_strings.translations.public_file_note;
                }

                var modal_content = `
                <div class="public_link_modal">
                    <p>` + file_title + `</p>
                    <div class="">
                        <textarea class="input-large form-control" rows="4" readonly>` + public_url + `</textarea>
                        <button class="public_link_copy btn btn-primary" data-copy-text="` + public_url + `">
                            <i class="fa fa-files-o" aria-hidden="true"></i> ` + json_strings.translations.click_to_copy + `
                        </button>
                    </div>
                    <p class="note">` + note_text + `</p>
                </div>`;

                Swal.fire({
                    title: json_strings.translations.public_url,
                    html: modal_content,
                    showCloseButton: false,
                    showCancelButton: false,
                    showConfirmButton: false,
                    showClass: {
                        popup: 'animate__animated animated__fast animate__fadeInUp'
                    },
                    hideClass: {
                        popup: 'animate__animated animated__fast animate__fadeOutDown'
                    }
                }).then((result) => {});
            });

            /** Used on the public link modal on both manage files and the upload results */
            $(document).on('click', '.public_link_copy', function(e) {
                var text = $(this).data('copy-text');
                copyTextToClipboard(text);
            });
        });
    };
})();
(function () {
    'use strict';

    admin.parts.select2 = function () {

        $(document).ready(function(){
            $('.select2').select2({
                width: '100%',
                theme: "bootstrap-5",
                allowClear: true,
            });

            $('.add-all').on('click', function() {
                var target = $(this).data('target');
                var selector = $('#'+target);
                $(selector).hide();
                $(selector).find('option').each(function() {
                    $(this).prop('selected', true);
                });
                $(selector).trigger('change');
                return false;
            });

            $('.remove-all').on('click', function() {
                var target = $(this).data('target');
                var selector = $('#'+target);
                selector.val(null).trigger('change');
                return false;
            });

            $('.copy-all').on('click', function() {
                if ( confirm( json_strings.translations.upload_form.copy_selection ) ) {
                    var target = $(this).data('target');
                    var type = $(this).data('type');
                    var selector = $('#'+target);
                    var val;
    
                    var selected = new Array();
                    $(selector).find('option:selected').each(function() {
                        selected.push($(this).val().toString());
                    });

                    $('.select2[data-type="'+type+'"]').not(selector).each(function() {
                        $(this).find('option').each(function() {
                            val = $(this).val().toString();
                            if (selected.includes(val)) {
                                $(this).prop('selected', 'selected');

                            } else {
                                $(this).removeAttr('selected');
                            }
                        });
                        $(this).trigger('change');
                    });
                }

                return false;
            });
        });
    };
})();
let sideModal = class {
    constructor()
    {
        let markup = `
            <div id="side_modal_cover"></div>

            <div id="side_modal" class="hidden" >
                <div id="side_modal_internal">
                    <div id="sm_header">
                        <span class="dismiss">
                            <span class="fa-stack fa-lg">
                                <i class="fa fa-circle-thin fa-stack-2x"></i>
                                <i class="fa fa-times fa-stack-1x"></i>
                            </span>
                        </span>
                        <div class="title">
                            <h5></h5>
                        </div>
                    </div>
                    <div class="slideDown loader" id="side_modal_loading_indicator">
                        Loading...
                    </div>
                    <div class="contentarea">
                        <div class="content"></div>
                    </div>
                </div>
            </div>
        `;

        document.body.insertAdjacentHTML('afterend', markup);

        this.triggers = document.querySelectorAll("[data-side-modal]");
        this.cover = document.getElementById('side_modal_cover');
        this.modal = document.getElementById('side_modal');
        this.header = document.getElementById('sm_header');
        this.closeButton = this.header.querySelector('.dismiss');
        this.titleEl = this.header.querySelector('h5');
        this.loader = this.modal.querySelector('.loader');
        this.contentArea = this.modal.querySelector('.content');
        this.isOpen = false;
        this.closeWithButtonOnly = false;
    }

    setUp() {
        document.addEventListener('click', this.shouldOpenSideModal.bind(this));

        this.triggers.forEach(trigger => {
            trigger.addEventListener("click", this.openSideModalAndLoadContent.bind(this));
        });

        this.cover.addEventListener("click", this.closeSideModalFromCover.bind(this));
        this.closeButton.addEventListener("click", this.closeSideModal.bind(this));

        document.addEventListener("keydown", this.closeWithEscKey.bind(this));
    }

    changeCloseWithButtonOnly(value)
    {
        if (typeof value == "boolean") {
            this.closeWithButtonOnly = value;
        }
    }

    closeWithEscKey(e)
    {
        if (e.keyCode === 27 && this.closeWithButtonOnly == false) {
            this.closeSideModal();
        }
    }

    closeSideModalFromCover()
    {
        if (this.closeWithButtonOnly == false) {
            this.closeSideModal();
        }
    }

    shouldOpenSideModal(e)
    {
        if (e.target && e.target.dataset.sideModal) {
            this.openSideModal();
        }
    }

    openSideModal(e)
    {
        this.isOpen = true;
        this.cover.classList.add('visible');
        this.modal.classList.remove('hidden');
        document.body.classList.add('ox-h');
    }

    openSideModalAndLoadContent(e)
    {
        e.preventDefault();
        this.isOpen = true;
        this.cover.classList.add('visible');
        this.modal.classList.remove('hidden');
        document.body.classList.add('ox-h');

        this.cleanAndSetLoading();
        this.setTitle(e.target.dataset.title);

        let url = e.target.href;
        let that = this;

        axios.get(url, {
        })
        .then(function (response) {
            that.setContent(response.data);
        })
        .catch(function (error) {
            toastr.error(json_strings.translations.cannot_load_content);

            that.setTitle('');
            that.setContent('');
            that.closeSideModal();
        });

    }

    closeSideModal()
    {
        if (this.isOpen) {
            this.isOpen = false;
            this.cover.classList.remove('visible');
            this.modal.classList.add('hidden');
            document.body.classList.remove('ox-h');

            this.clean();
        }
    }

    setTitle(title)
    {
        this.titleEl.innerHTML = title;
    }

    setContent(content)
    {
        this.contentArea.innerHTML = content;
        this.loader.classList.remove('visible');
    }

    clean()
    {
        this.contentArea.innerHTML = "";
    }

    cleanAndSetLoading()
    {
        this.clean();
        this.loader.classList.add('visible');
    }
};

window.smd = new sideModal();
window.smd.setUp();

(function () {
    'use strict';
    
    admin.parts.widgetActionLog = function () {
        
        $(document).ready(function(){
            // Action log
            function ajax_widget_log( action ) {
                var target = $('#log_container');
                var select = $('#widget_actions_log_change');
                var list = $('<ul/>').addClass('none');

                target.html('');
                select.attr('disabled', 'disabled');
                $('#widget_actions_log .loading-icon').removeClass('none');

                $.ajax({
                    url: json_strings.uri.widgets+'ajax/actions-log.php',
                    data: { action:action },
                    cache: false,
                }).done(function(data) {
                    var obj = JSON.parse(data);
                    //console.log(obj);
                    $.each(obj.actions, function(i, item) {
                        var line = [];
                        var parts = {
                            part1: item.part1,
                            action: item.action,
                            part2: item.part2,
                            part3: item.part3,
                            part4: item.part4,
                        }

                        for (const key in parts) {
                            if (parts[key]) {
                                line.push('<span class="item_'+key+'">'+parts[key]+'</span>');
                            }
                        };

                        var icon;
                        switch (item.type) {
                            case 'system': icon = 'cog'; break;
                            case 'auth': icon = 'lock'; break;
                            case 'files': icon = 'file'; break;
                            case 'clients': icon = 'address-card'; break;
                            case 'users': icon = 'users'; break;
                            case 'groups': icon = 'th-large'; break;
                            case 'categories': icon = 'object-group'; break;
                            default: icon = 'cog'; break;
                        }
                        line = line.join(' ');
                        var li = $('<li/>')
                            .appendTo(list)
                            .html(`
                                <div class="date">
                                    <span>`+
                                        item.timestamp+`
                                    </span>
                                    <i class="fa fa-`+icon+`" aria-hidden="true"></i>
                                </div>
                                <div class="action">`+
                                    htmlEncode(item.formatted)+`
                                </div>
                            `);
                    });
                    //console.log(list);
                    target.append(list);
                    list.slideDown();
                }).fail(function(data) {
                    target.html(json_strings.translations.failed_loading_resource);
                }).always(function() {
                    $('#widget_actions_log .loading-icon').addClass('none');
                });

                $(select).removeAttr('disabled');
            }

            // Action log
            $('#widget_actions_log_change').on('change', function(e) {
                var action = $('#widget_actions_log_change option').filter(':selected').val()
                ajax_widget_log(action);
            });

			ajax_widget_log();
        });
    };
})();
(function () {
    'use strict';
    
    admin.parts.widgetNews = function () {
        
        $(document).ready(function(){
            // Action log
            function ajax_widget_news( action ) {
                var target = $('#news_container');
                var list = $('<ul/>').addClass('none home_news list-unstyled');

                target.html('');
                $('#widget_projectsend_news .loading-icon').removeClass('none');

                $.ajax({
                    url: json_strings.uri.widgets+'ajax/news.php',
                    cache: false,
                }).done(function(data) {
                    // var obj = JSON.parse(data);
                    var obj = data;
                    $.each(obj.items, function(i, item) {
                        var li = $('<li/>')
                            .appendTo(list)
                            .html(`
                                <span class="date">`+item.date+`</span>
                                    <a href="`+item.url+`" target="_blank">
                                        <h5>`+item.title+`</h5>
                                    </a>
                                <p>`+item.content+`</p>
                            `);
                    });
                    //console.log(list);
                    target.append(list);
                    list.slideDown();
                }).fail(function(data) {
                    target.html(json_strings.translations.failed_loading_resource);
                }).always(function() {
                    $('#widget_projectsend_news .loading-icon').addClass('none');
                });
            }

			ajax_widget_news();
        });
    };
})();

(function () {
    'use strict';
    
    admin.parts.widgetStatistics = function () {
        
        $(document).ready(function(){
            var chart;

            // Get current theme colors
            function getThemeColors() {
                var isDarkMode = document.documentElement.getAttribute('data-theme') === 'dark';
                return {
                    textColor: isDarkMode ? '#e9ecef' : '#333333',
                    gridColor: isDarkMode ? '#4a5568' : '#e5e5e5',
                    tooltipBg: isDarkMode ? '#2d3748' : '#fff'
                };
            }

            // Update chart colors for current theme
            function updateChartColors() {
                if (!chart) return;

                var colors = getThemeColors();

                // Update legend
                if (chart.options.legend && chart.options.legend.labels) {
                    chart.options.legend.labels.fontColor = colors.textColor;
                }

                // Update scales
                if (chart.options.scales) {
                    if (chart.options.scales.xAxes) {
                        chart.options.scales.xAxes.forEach(axis => {
                            if (axis.ticks) axis.ticks.fontColor = colors.textColor;
                            if (axis.gridLines) axis.gridLines.color = colors.gridColor;
                        });
                    }
                    if (chart.options.scales.yAxes) {
                        chart.options.scales.yAxes.forEach(axis => {
                            if (axis.ticks) axis.ticks.fontColor = colors.textColor;
                            if (axis.gridLines) axis.gridLines.color = colors.gridColor;
                        });
                    }
                }

                // Update tooltips
                if (chart.options.tooltips) {
                    chart.options.tooltips.backgroundColor = colors.tooltipBg;
                    chart.options.tooltips.titleFontColor = colors.textColor;
                    chart.options.tooltips.bodyFontColor = colors.textColor;
                    chart.options.tooltips.borderColor = colors.gridColor;
                }

                chart.update();
            }

            // Listen for theme changes
            document.addEventListener('themechange', function() {
                updateChartColors();
            });

            // Statistics chart
            function ajax_widget_statistics(days) {
                var _chart_container = $('#widget_statistics #chart_container');
                _chart_container.find('canvas').remove();
                $('#widget_statistics .loading-icon').removeClass('none');
                if (chart) {
                    chart.destroy();
                }
                $.ajax({
                    url: json_strings.uri.widgets+'ajax/statistics.php',
                    data: { days:days },
                    cache: false,
                }).done(function(data) {
                    // var obj = JSON.parse(data);
                    var obj = data;
                    _chart_container.append('<canvas id="chart_statistics"><canvas>');

                    // Get theme colors using centralized function
                    var colors = getThemeColors();

                    chart = new Chart(document.getElementById('chart_statistics'), {
                        type: 'line',
                        data: obj.chart,
                        options: {
                            responsive: true,
                            title: {
                                display: false
                            },
                            tooltips: {
                                mode: 'index',
                                intersect: false,
                                backgroundColor: colors.tooltipBg,
                                titleFontColor: colors.textColor,
                                bodyFontColor: colors.textColor,
                                borderColor: colors.gridColor,
                                borderWidth: 1
                            },
                            legend: {
                                labels: {
                                    fontColor: colors.textColor
                                }
                            },
                            scales: {
                                xAxes: [{
                                    display: true,
                                    ticks: {
                                        fontColor: colors.textColor
                                    },
                                    gridLines: {
                                        color: colors.gridColor
                                    }
                                }],
                                yAxes: [{
                                    display: true,
                                    ticks: {
                                        fontColor: colors.textColor
                                    },
                                    gridLines: {
                                        color: colors.gridColor
                                    }
                                }]
                            },
                            elements: {
                                line: {
                                    tension: 0
                                }
                            }
                        }
                    });
                }).fail(function(data) {
                    _chart_container.html(json_strings.translations.failed_loading_resource);
                }).always(function() {
                    $('#widget_statistics .loading-icon').addClass('none');
                });
    
                return;
            }

            // Statistics
            $('#widget_statistics button.get_statistics').on('click', function(e) {
                if ($(this).hasClass('active')) {
                    return false;
                }
                else {
                    var days = $(this).data('days');
                    $('#widget_statistics button.get_statistics').removeClass('btn-primary active').addClass('btn-pslight');
                    $(this).addClass('btn-primary active').removeClass('btn-pslight');
                    ajax_widget_statistics(days);
                }
            });

			ajax_widget_statistics(15);
        });
    };
})();
//# sourceMappingURL=app.js.map
